package com.xebialabs.xlrelease.runner.api.v1

import com.fasterxml.jackson.core.`type`.TypeReference
import com.fasterxml.jackson.module.scala.JsonScalaEnumeration
import com.xebialabs.xlrelease.runner.api.v1.JobRunnerResource.ReleaseServerStates.ReleaseServerState
import com.xebialabs.xlrelease.runner.api.v1.JobRunnerResource.{ConfirmationResponse, HealthResponse, JobReservationResponse, LogEntryRequest}
import com.xebialabs.xlrelease.runner.domain.{JobData, JobDirective, JobId, JobResult}

import javax.ws.rs._
import javax.ws.rs.core.{Context, MediaType}
import javax.ws.rs.sse.{Sse, SseEventSink}

/**
 * This is internal API for runners and should not be documented.
 */
@Path("/api/v1/runners")
@Consumes(Array(MediaType.APPLICATION_JSON))
@Produces(Array(MediaType.APPLICATION_JSON))
trait JobRunnerResource {

  /**
   * Reserves a job for execution by given runner id.
   *
   * @param runnerId  runner id
   * @param requestId message request id
   * @return response with Some(job data) of the reserved job or None if nothing was reserved
   */
  @GET
  @Path("{runnerId:.+}/job")
  def reserveJob(@PathParam("runnerId") runnerId: String, @QueryParam("requestId") requestId: String): JobReservationResponse

  /**
   * Confirms job reservation before it can be executed by a runner
   * and in return receives information if confirmation was successful.
   *
   * @param runnerId  id of the runner that confirms reservation
   * @param jobId     id of the job to be executed
   * @param requestId message request id
   * @return response with confirmation result
   */
  @PUT
  @Path("{runnerId:.+}/job/{jobId:.+}")
  def confirmJob(@PathParam("runnerId") runnerId: String, @PathParam("jobId") jobId: JobId, @QueryParam("requestId") requestId: String): ConfirmationResponse

  @POST
  @Path("{runnerId:.+}/jobs/result")
  def finishJob(@PathParam("runnerId") runnerId: String, @QueryParam("requestId") requestId: String, jobResult: JobResult): ConfirmationResponse

  @POST
  @Path("{runnerId:.+}/log")
  def log(@PathParam("runnerId") runnerId: String, @QueryParam("requestId") requestId: String, logEntryRequest: LogEntryRequest): ConfirmationResponse

  @POST
  @Path("{runnerId:.+}/directives")
  def executeDirectives(@PathParam("runnerId") runnerId: String,
                        @QueryParam("requestId") requestId: String,
                        directives: Seq[JobDirective]): ConfirmationResponse

  @GET
  @Path("{runnerId:.+}/commands")
  @Produces(Array(MediaType.SERVER_SENT_EVENTS))
  def commands(@PathParam("runnerId") runnerId: String, @Context sink: SseEventSink, @Context sse: Sse): Unit

  @POST
  @Path("{runnerId:.+}/commands/{commandId:.+}/confirm")
  def confirmCommand(@PathParam("runnerId") runnerId: String, @PathParam("commandId") commandId: String): ConfirmationResponse

  @GET
  @Path("{runnerId:.+}/health")
  def health(@PathParam("runnerId") runnerId: String, @QueryParam("requestId") requestId: String): HealthResponse
}

object JobRunnerResource {
  sealed trait JobResourceResponse {
    def requestId: String
  }

  class ReleaseServerStatesType extends TypeReference[ReleaseServerStates.type]

  case class HealthResponse(requestId: String,
                            @JsonScalaEnumeration(classOf[ReleaseServerStatesType]) serverState: ReleaseServerState)
    extends JobResourceResponse

  object ReleaseServerStates extends Enumeration {
    type ReleaseServerState = Value
    val STARTING: ReleaseServerState = Value("STARTING")
    val RUNNING: ReleaseServerState = Value("RUNNING")
    val MAINTENANCE: ReleaseServerState = Value("MAINTENANCE")
  }

  case class JobReservationResponse(requestId: String, job: Option[JobData]) extends JobResourceResponse

  case class ConfirmationResponse(requestId: String, isConfirmed: Boolean) extends JobResourceResponse

  case class LogEntryRequest(taskId: String, executionId: String, jobId: Long, chunk: Long, lastEntryTimestamp: String, payload: Array[Byte], uriScheme: String)

}
