package com.xebialabs.xlrelease.planner

import org.joda.time.DateTime

case class PlannerState private(now: DateTime,
                                stack: List[String],
                                plannedIds: Set[String],
                                releaseTreeById: Map[String, PlannerReleaseItem]) {

  def addReleaseId(releaseId: String): PlannerState = this.copy(plannedIds = plannedIds + releaseId)

  def updated(plan: Plans.Computed): PlannerState = {
    releaseTreeById.get(plan.item.id).foreach { item =>
      item.setStartDate(plan.start)
      item.setEndDate(plan.end)
    }
    this
  }

  def find(id: String): Option[PlannerReleaseItem] = releaseTreeById.get(id)

  def isPlanned(id: String): Boolean = plannedIds.contains(id)
}

object PlannerState {
  type S = PlannerState
  type ComputeTransition = State[S, Plans.Computed]

  def apply(releaseTree: PlannerReleaseTree, now: DateTime): Option[PlannerState] = {
    flattenReleaseTree(releaseTree).map { releaseTreeById =>
      PlannerState(
        now = now,
        stack = List.empty,
        plannedIds = Set.empty,
        releaseTreeById = releaseTreeById
      )
    }
  }

  def flattenReleaseTree(releaseTree: PlannerReleaseTree): Option[Map[String, PlannerReleaseItem]] = {
    mergeReleaseMaps(
      (releaseTree.root :: releaseTree.dependentReleases).map(flattenRelease _)
    )
  }

  def flattenRelease(releaseItem: PlannerReleaseItem): Option[Map[String, PlannerReleaseItem]] = {
    val all = flattenPlan(releaseItem)
    val grouped = all.groupBy(_.id)
    if (grouped.keySet.size != all.size) {
      // non-unique IDs
      None
    } else {
      Some(
        grouped
          .view
          .mapValues(_.headOption)
          .toMap
          .collect {
            case (id, Some(item)) => (id, item)
          }
      )
    }
  }

  def mergeReleaseMaps(maps: List[Option[Map[String, PlannerReleaseItem]]]): Option[Map[String, PlannerReleaseItem]] = {
    maps.foldLeft(Option.apply(Map.empty[String, PlannerReleaseItem])) {
      case (Some(merged), Some(map)) => {
        if (map.forall { case (k, v) => merged.get(k).map(_ == v).getOrElse(true) }) {
          // either unique ID or pointing to same object.
          Some(merged ++ map)
        } else {
          // duplicate ID (pointing to different objects)
          None
        }
      }
      case _ => None
    }
  }

  def flattenPlan(treeItem: PlannerReleaseItem): List[PlannerReleaseItem] = treeItem :: treeItem.allChildren

  def state: State[S, S] = State.get[S]

  def enter(itemId: String): State[S, Unit] = State.update(s => s.copy(stack = itemId :: s.stack))

  def exit: State[S, Unit] = State.update(s => s.copy(stack = s.stack.headOption.map(_ => s.stack.tail).getOrElse(Nil)))

  def update(plan: Plans.Computed): State[S, Unit] = State.update(_.updated(plan))
}
