package com.xebialabs.xlrelease.planner

import com.xebialabs.xlrelease.domain.status.TaskStatus

sealed trait PlannerReleaseItemType {
  def children: List[PlannerReleaseItem]
  def dependencies: List[PlannerDependency]
  def links: List[PlannerLink]
}

object PlannerReleaseItemType {

  case class Release(phases: List[PlannerReleaseItem]) extends PlannerReleaseItemType {
    def children = phases
    def dependencies = List.empty
    def links = List.empty
  }

  case class Phase(tasks: List[PlannerReleaseItem]) extends PlannerReleaseItemType {
    def children = tasks
    def dependencies = List.empty
    def links = List.empty
  }

  case class Task(taskType: TaskType, taskStatus: TaskStatus, active: Boolean, automated: Boolean) extends PlannerReleaseItemType {
    def children = taskType.children
    def dependencies = taskType.dependencies
    def links = taskType.links

    def isGroup = taskType match {
      case (_: TaskType.Group) => true
      case _ => false
    }

    def isGate = taskType match {
      case (_: TaskType.Gate) => true
      case _ => false
    }

    def isParallelGroup = taskType match {
      case (_: TaskType.ParallelGroup) => true
      case _ => false
    }

    def isManual: Boolean = !isGroup && !automated

    def isDependent: Boolean = taskType match {
      case TaskType.Gate(deps) => !deps.isEmpty
      case _ => false
    }

    def isDoneInAdvance: Boolean = taskStatus match {
      case TaskStatus.COMPLETED_IN_ADVANCE | TaskStatus.SKIPPED_IN_ADVANCE => true
      case _ => false
    }

    def isActiveLeaf: Boolean = !isGroup && active
  }

  sealed trait TaskType {
    def children: List[PlannerReleaseItem]
    def dependencies: List[PlannerDependency]
    def links: List[PlannerLink]
  }
  object TaskType {
    sealed trait Group extends TaskType {
      def tasks: List[PlannerReleaseItem]
      def children = tasks
      def dependencies = List.empty
    }
    case class SequentialGroup(tasks: List[PlannerReleaseItem]) extends Group {
      def links = List.empty
    }
    case class ParallelGroup(tasks: List[PlannerReleaseItem], links: List[PlannerLink]) extends Group
    case class Gate(dependencies: List[PlannerDependency]) extends TaskType {
      def children = List.empty
      def links = List.empty
    }
    case object Other extends TaskType {
      def children = List.empty
      def links = List.empty
      def dependencies = List.empty
    }
  }
}
