package com.xebialabs.xlrelease.planner

import com.xebialabs.xlrelease.domain.PlanItem
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.utils.Graph
import org.joda.time.DateTime

object Planner {
  val MANUAL_TASK_DURATION = 60
  val AUTOMATED_TASK_DURATION = 1
  val FROM_END_TO_START = false

  val ALWAYS_USE_SCHEDULED_DATES: PlannerReleaseItem => Boolean = _ => true

  def makePlan(planItem: PlanItem, now: DateTime = DateTime.now()): Option[Plans.Computed] = {
    makePlan(PlannerReleaseTree(PlannerReleaseItem.transform(planItem)), now, ALWAYS_USE_SCHEDULED_DATES)
  }

  def makePlan(releaseTree: PlannerReleaseTree): Option[Plans.Computed] = makePlan(releaseTree, DateTime.now)

  def makePlan(releaseTree: PlannerReleaseTree, now: DateTime): Option[Plans.Computed] = makePlan(releaseTree, now, ALWAYS_USE_SCHEDULED_DATES)

  def makePlan(releaseTree: PlannerReleaseTree, now: DateTime, useScheduledDates: PlannerReleaseItem => Boolean): Option[Plans.Computed] =
    MakePlan.apply(releaseTree, now, useScheduledDates).apply

  private case class MakePlan(releaseTree: PlannerReleaseTree, now: DateTime, useScheduledDates: PlannerReleaseItem => Boolean) {
    def apply: Option[Plans.Computed] = {
      PlannerState(releaseTree, now).map(initialState =>
        makePlan0(releaseTree.root, proposedStartDate = initialState.now, parent = None).run0(initialState)
      )
    }

    import PlannerState._

    def makePlan0(item: PlannerReleaseItem, proposedStartDate: DateTime, parent: Option[PlannerReleaseItem]): ComputeTransition = {
      val plan = Plans.Base(item, start = computeStartDate(item, proposedStartDate, useScheduledDates(item)))
      for {
        s <- state
        computed <- {
          if (s.stack contains item.id) {
            cycle(plan)
          } else {
            for {
              _ <- enter(item.id)
              computedInner <- {
                if (item.isParallelGroup) {
                  parallel(plan)
                } else if (item.isGateTask) {
                  gate(plan)
                } else {
                  container(plan)
                }
              }
              _ <- exit
            } yield computedInner
          }
        }
        _ <- update(computed)
      } yield computed
    }

    def cycle(plan: Plans.Planned): ComputeTransition = State.const(plan.circular)

    def parallel(plan: Plans.Planned): ComputeTransition = {
      for {
        children <- makeSubTaskPlans(plan.item, plan.start, parent = Some(plan.item))
        now <- state.map(_.now)
      } yield {
        val computedEndDate = computeEndDate(plan.item, findMaxDate(plan.start, children.map(_.end)), plan.start, now, useScheduledDates(plan.item))
        plan.container(computedEndDate, children)
      }
    }

    def gate(plan: Plans.Planned): ComputeTransition = {
      for {
        dependenciesEndDate <- computeDependenciesEndDate(plan, plan.item)
        now <- state.map(_.now)
      } yield {
        plan.normal(computeEndDate(plan.item, dependenciesEndDate, plan.start, now, useScheduledDates(plan.item)))
      }
    }

    def container(plan: Plans.Planned): ComputeTransition = {
      state.map { s =>
        plan.item.children
          .map(computeChild(plan.item))
          .sequence
          .flatMap { children =>
            State.get.map { case (s1, lastChildDate) =>
              plan.container(computeEndDate(plan.item, lastChildDate, plan.start, s1.now, useScheduledDates(plan.item)), children)
            }
          }.run0((s, plan.start))
      }
    }

    def computeChild(parent: PlannerReleaseItem)(child: PlannerReleaseItem): State[(S, DateTime), Plans.Computed] = {
      State { case (s, currentDate) =>
        makePlan0(child, currentDate, parent = Some(parent)).map { childPlan =>
          val newCurrentDate = {
            if (!child.isTaskDoneInAdvance) {
              childPlan.end
            } else {
              currentDate
            }
          }
          childPlan -> newCurrentDate
        }.run(s) match {
          case (s1, (childPlan, newCurrentDate)) =>
            (s1, newCurrentDate) -> childPlan
        }
      }
    }

    def computeDependenciesEndDate(plan: Plans.Planned, item: PlannerReleaseItem): State[S, DateTime] = {
      if (!item.hasOwnEndDate && !item.hasPlannedDuration) {
        item.dependencies
          .filter(_.isDependencyUnresolved)
          .map(computeDependencyEndDate) // List[State[S, Option[DateTime]]]
          .sequence // State[S, List[Option[DateTime]]
          .map { dependenciesOptDates: List[Option[DateTime]] =>
          findMaxDate(plan.start, dependenciesOptDates.flatten)
        } // State[S, DateTime]
      } else {
        State.const(plan.start) // State[S, DateTime]
      }
    }

    def computeDependencyEndDate(dependency: PlannerDependency): State[S, Option[DateTime]] = {
      dependency.ownEndDate.map(endDate => State.const[S, Option[DateTime]](Some(endDate))).getOrElse {
        state.flatMap { s =>
          val maybeGetDate: Option[State[S, Option[DateTime]]] = {
            for {
              dependencyTarget <- dependency.target
              target <- s.find(dependencyTarget.id)
              targetRelease <- s.find(Ids.releaseIdFrom(dependencyTarget.id))
            } yield {
              if (s.isPlanned(targetRelease.id)) {
                State.const(target.endDate)
              } else {
                for {
                  _ <- makePlan0(targetRelease, s.now, parent = None)
                  s1 <- state
                } yield s1.find(target.id).flatMap(_.endDate)
              }
            }
          }
          maybeGetDate.getOrElse(State.const(Option.empty[DateTime]))
        }
      }
    }

    def makeSubTaskPlans(item: PlannerReleaseItem, currentDate: DateTime, parent: Option[PlannerReleaseItem]): State[S, List[Plans.Computed]] = {
      val g = Graph(item.links.map(link => Graph.Edge(link.source.id -> link.target.id)))
      val scope = item.children.map(child => child.id -> child).toMap

      if (!g.hasCycle) {
        val all = item.children.map(_.id).toSet
        val order = (all diff g.nodes).toList ++ g.order
        order.map { subItemId =>
          val subItem = scope(subItemId)
          val predecessors = g.incoming(subItemId).toList.flatMap(scope.get)
          val startDate = findMaxDate(currentDate, predecessors.flatMap(_.endDate))
          state.flatMap { _ =>
            makePlan0(subItem, startDate, parent = parent)
          }
        }.sequence
      } else {
        State.const(List.empty[Plans.Computed])
      }
    }
  }

  def computeStartDate(item: PlannerReleaseItem, proposedStartDate: DateTime, useScheduledDates: Boolean): DateTime = {
    startOfMinute(
      item.startDate
        .orElse(if (useScheduledDates) item.scheduledStartDate else None)
        .orElse {
          if (useScheduledDates) {
            for {
              dueDate <- item.dueDate
              duration <- item.plannedDuration
            } yield dueDate.minus(duration)
          } else {
            None
          }
        }
        .getOrElse(proposedStartDate)
    )
  }

  def computeEndDate(item: PlannerReleaseItem, proposedDate: DateTime, displayStartDate: DateTime, now: DateTime, useScheduledDates: Boolean): DateTime = {
    val endDate = item.endDate
      .orElse(if (useScheduledDates) item.dueDate else None)
      .orElse(
        if (useScheduledDates) {
          item.plannedDuration.map { plannedDuration =>
            displayStartDate.plus(plannedDuration)
          }
        } else {
          None
        }
      )
      .orElse(
        if (item.isAutomatedTask) {
          Some(proposedDate.plusMinutes(AUTOMATED_TASK_DURATION))
        } else {
          None
        }
      )
      .orElse(
        if (item.isManualTask && !item.isDependentTask) {
          Some(proposedDate.plusMinutes(MANUAL_TASK_DURATION))
        } else {
          None
        }
      )
      .getOrElse(proposedDate)

    val endDate1 = latest(displayStartDate, endDate)
    val endDate2 = if (item.isActiveLeafTask) latest(now, endDate1) else endDate1

    startOfMinute(endDate2)
  }

  def startOfMinute(dt: DateTime): DateTime = dt.withSecondOfMinute(0).withMillisOfSecond(0) //dt.withSecondsOfMinute(0).withMillisOfSecond(0)

  def latest(a: DateTime, b: DateTime): DateTime = if (a.isAfter(b)) a else b

  def findMaxDate(default: DateTime, dates: List[DateTime]): DateTime = dates.foldLeft(default)(latest)

  def findPredecessorsIds(links: List[PlannerLink], itemId: String): Set[String] = {
    findPredecessorsIds0(links, itemId)(List.empty, Set.empty)
  }

  @scala.annotation.tailrec
  def findPredecessorsIds0(links: List[PlannerLink], itemId: String)(toProcess: List[String], found: Set[String]): Set[String] = {
    val nextIds: List[String] = links.filter(_.target.id == itemId).map(_.source.id)
    toProcess ++ nextIds match {
      case Nil => found
      case nextId :: rest => findPredecessorsIds0(links, nextId)(toProcess ++ rest, found ++ nextIds)
    }
  }

  def mapIdsToItems(ids: List[String], items: List[PlannerReleaseItem]): List[Option[PlannerReleaseItem]] = ids.map(id => items.find(_.id == id))
}
