package com.xebialabs.xlrelease.planner

import com.xebialabs.xlrelease.domain._
import org.joda.time.{DateTime, Duration}

import java.util.{Date, Objects}
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

//noinspection ScalaStyle
case class PlannerReleaseItem(item: PlanItem, var itemType: PlannerReleaseItemType) {
  var currentStartDate = item.getStartDate

  var currentEndDate = item.getEndDate

  def getItem(): PlanItem = item

  def getRelease(): Option[Release] = Option(getItem()).filter(_ => isRelease).map(_.asInstanceOf[Release])

  def getPhases(): java.util.List[PlannerReleaseItem] = children.filter(_.isPhase).asJava

  def getTask: Option[Task] = Option(getItem()).filter(_ => isTask).map(_.asInstanceOf[Task])

  def getTasks(): java.util.List[PlannerReleaseItem] = children.filter(_.isTask).asJava

  def getAllTasks(): java.util.List[PlannerReleaseItem] = allChildren.filter(_.isTask).asJava

  def getStartDate: Date = startDate.map(_.toDate).orNull

  def getEndDate: Date = endDate.map(_.toDate).orNull

  def getScheduledStartDate: Date = scheduledStartDate.map(_.toDate).orNull

  def getDueDate: Date = dueDate.map(_.toDate).orNull

  def getPlannedDurationInMinutes: Long = plannedDuration.map(_.getStandardMinutes).getOrElse(-1L)

  def getDuration: Duration = plannedDuration.orElse {
    for {
      end <- endDate.orElse(dueDate)
      start <- startDate.orElse(scheduledStartDate)
    } yield new Duration(start, end)
  }.orNull

  def getComputedPlannedDuration: Duration = item.getComputedPlannedDuration

  def getActualDuration: Duration = item.getActualDuration

  def id: String = item.getId

  def startDate: Option[DateTime] = currentStartDate

  def endDate: Option[DateTime] = currentEndDate

  def scheduledStartDate: Option[DateTime] = item.getScheduledStartDate

  def dueDate: Option[DateTime] = item.getDueDate

  def plannedDuration: Option[Duration] = item.getPlannedDuration

  def children: List[PlannerReleaseItem] = itemType.children

  def allChildren: List[PlannerReleaseItem] = itemType.children.flatMap(c => c :: c.allChildren)

  def dependencies: List[PlannerDependency] = itemType.dependencies

  def links: List[PlannerLink] = itemType.links

  def hasOwnEndDate: Boolean = endDate.orElse(dueDate).isDefined

  def hasPlannedDuration: Boolean = plannedDuration.isDefined

  def isManualTask: Boolean = ifTask(_.isManual)

  def isDependentTask: Boolean = ifTask(_.isDependent)

  def isTaskDoneInAdvance: Boolean = ifTask(_.isDoneInAdvance)

  def isAutomatedTask: Boolean = ifTask(_.automated)

  def isActiveLeafTask: Boolean = ifTask(_.isActiveLeaf)

  def isParallelGroup: Boolean = ifTask(_.isParallelGroup)

  def isGateTask: Boolean = ifTask(_.isGate)

  def setStartDate(date: DateTime): Unit = currentStartDate = date.toDate

  def setEndDate(date: DateTime): Unit = currentEndDate = date.toDate

  def isTask: Boolean = ifTask(_ => true)

  def ifTask(predicate: PlannerReleaseItemType.Task => Boolean): Boolean = itemType match {
    case (task: PlannerReleaseItemType.Task) => predicate(task)
    case _ => false
  }

  def isPhase: Boolean = itemType match {
    case (_: PlannerReleaseItemType.Phase) => true
    case _ => false
  }

  def isRelease: Boolean = itemType match {
    case (_: PlannerReleaseItemType.Release) => true
    case _ => false
  }

  def canEqual(that: Any): Boolean = that.isInstanceOf[PlannerReleaseItem]

  override def equals(that: Any): Boolean = that match {
    case that: PlannerReleaseItem => that.canEqual(this) && Objects.equals(this.id, that.id);
    case _ => false
  }

  override def hashCode: Int = Objects.hashCode(id)

  implicit def dateToOptJoda(nullableJavaDate: Date): Option[DateTime] = Option(nullableJavaDate).map(new DateTime(_))

  implicit def durationToOptJoda(nullableJavaDuration: Integer): Option[Duration] = Option(nullableJavaDuration).map(d => Duration.standardSeconds(d.toLong))
}

object PlannerReleaseItem {
  implicit def transform(item: PlanItem): PlannerReleaseItem = {
    transform(item, mutable.Map.empty)
  }

  def transform(item: PlanItem, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerReleaseItem = {
    val cachedItem = transformedItems.get(item.getId)
    if (cachedItem.isDefined) cachedItem.get else transform0(item, transformedItems)
  }

  def transform0(item: PlanItem, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerReleaseItem = {
    val currentItem = PlannerReleaseItem(item, null)
    transformedItems.put(item.getId, currentItem)
    val transformedItem = item match {
      case (r: Release) => transformRelease(r, transformedItems)
      case (p: Phase) => transformPhase(p, transformedItems)
      case (t: Task) => transformTask(t, transformedItems)
      case _ => throw new IllegalStateException(s"Unable to match $item")

    }
    currentItem.itemType = transformedItem.itemType
    currentItem
  }

  def transformRelease(r: Release, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerReleaseItem = {
    PlannerReleaseItem(r, PlannerReleaseItemType.Release(phases = r.getPhases.asScala.toList.map(transformPhase(_, transformedItems))))
  }

  def transformPhase(p: Phase, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerReleaseItem = {
    PlannerReleaseItem(p, PlannerReleaseItemType.Phase(tasks = p.getTasks.asScala.toList.map(transformTask(_, transformedItems))))
  }

  def transformTask(t: Task, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerReleaseItem = {
    PlannerReleaseItem(t, PlannerReleaseItemType.Task(taskType(t, transformedItems), taskStatus = t.getStatus, active = t.isActive, automated = t.isAutomated))
  }

  def taskType(t: Task, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerReleaseItemType.TaskType =
    t match {
      case (gate: GateTask) =>
        PlannerReleaseItemType.TaskType.Gate(gate.getDependencies.asScala.toList.map(transformDependency(_, transformedItems)))

      case (sequential: SequentialGroup) =>
        PlannerReleaseItemType.TaskType.SequentialGroup(sequential.getTasks.asScala.toList.map(transformTask(_, transformedItems)))

      case (parallel: ParallelGroup) =>
        PlannerReleaseItemType.TaskType.ParallelGroup(
          parallel.getTasks.asScala.toList.map(transformTask(_, transformedItems)),
          parallel.getLinks.asScala.toList.map(transformLink(_, transformedItems))
        )

      case _ =>
        PlannerReleaseItemType.TaskType.Other
    }

  def transformDependency(dependency: Dependency, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerDependency = {
    val targetOpt = Option(dependency.getTarget[PlanItem]).map(target => transformedItems.getOrElse(target.getId, transform(target, transformedItems)))
    PlannerDependency(dependency.getId, dependency.isDone, dependency.isArchived, targetOpt)
  }

  def transformLink(link: Link, transformedItems: mutable.Map[String, PlannerReleaseItem]): PlannerLink = {
    val source = transformedItems.getOrElse(link.getSource.getId, transform(link.getSource, transformedItems))
    val target = transformedItems.getOrElse(link.getTarget.getId, transform(link.getTarget, transformedItems))
    PlannerLink(source, target)
  }
}
