package com.xebialabs.xlrelease.serialization.json.xltype;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import jakarta.json.stream.JsonGenerator;
import org.joda.time.DateTime;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.udm.ExternalProperty;
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xlrelease.serialization.json.utils.JsonWithCachedProvider;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.util.DateUtil;

public class CiJson2Writer implements CiWriter, AutoCloseable {

    protected final Writer writer;
    protected final JsonGenerator json;

    public CiJson2Writer() {
        this(new StringWriter());
    }

    public CiJson2Writer(Writer writer) {
        this.writer = writer;
        this.json = JsonWithCachedProvider.createGenerator(writer);
    }

    public Writer getWriter() {
        return writer;
    }

    @Override
    public String toString() {
        return writer.toString();
    }

    @Override
    public void startList() {
        json.writeStartArray();
    }

    @Override
    public void endList() {
        json.writeEnd();
        json.flush();
    }

    @Override
    public void startCi(String type, String id) {
        json.writeStartObject();
        if (null == id) {
            json.writeNull("id");
        } else {
            json.write("id", id);
        }
        json.write("type", type);
    }

    @Override
    public void endCi() {
        json.writeEnd();
        json.flush();
    }

    @Override
    public void token(String token) {
        json.write("$token", token != null ? token : "");
    }


    @Override
    public void ciAttributes(CiAttributes ciAttributes) {
        addStringAttribute("$createdBy", ciAttributes.getCreatedBy());
        addDateAttribute("$createdAt", ciAttributes.getCreatedAt());
        addStringAttribute("$lastModifiedBy", ciAttributes.getLastModifiedBy());
        addDateAttribute("$lastModifiedAt", ciAttributes.getLastModifiedAt());
        String scmTraceabilityDataInt = ciAttributes.getScmTraceabilityDataId();
        String scmTraceabilityData = null;
        if (scmTraceabilityDataInt != null) {
            scmTraceabilityData = scmTraceabilityDataInt;
        }
        addStringAttribute("$scmTraceabilityDataId", scmTraceabilityData);
    }

    @Override
    public void ciFileAttribute(String file) {
        addStringAttribute("file", file);
    }

    private void addStringAttribute(String attrName, String attrValue) {
        if (attrValue != null) {
            json.write(attrName, attrValue);
        }
    }

    private void addDateAttribute(String attrName, DateTime attrValue) {
        if (attrValue != null) {
            json.write(attrName, DateUtil.toString(attrValue));
        }
    }

    @Override
    public void startProperty(String name) {
        json.writeKey(name);
    }

    @Override
    public void endProperty() {
        json.flush();
    }

    @Override
    public void valueAsString(Object value) {
        if (null == value) {
            json.writeNull();
        } else {
            if (value instanceof Integer) {
                json.write((Integer) value);
            } else if (value instanceof Long) {
                json.write((Long) value);
            } else if (value instanceof Byte) {
                json.write((Byte) value);
            } else if (value instanceof Short) {
                json.write((Short) value);
            } else if (value instanceof Double) {
                json.write((Double) value);
            } else if (value instanceof Float) {
                json.write((Float) value);
            } else if (value instanceof Boolean) {
                json.write((Boolean) value);
            } else if (value instanceof BigDecimal) {
                json.write((BigDecimal) value);
            } else if (value instanceof BigInteger) {
                json.write((BigInteger) value);
            } else if (value instanceof String) {
                json.write((String) value);
            } else {
                json.write(value.toString());
            }
        }
    }

    @Override
    public void valuesAsStrings(Collection<?> values) {
        json.writeStartArray();
        for (Object item : values) {
            valueAsString(item);
        }
        json.writeEnd();
    }

    @Override
    public void mapAsStrings(Map<?, ?> map) {
        json.writeStartObject();
        for (Object key : map.keySet()) {
            json.writeKey(String.valueOf(key));
            valueAsString(map.get(key));
        }
        json.writeEnd();
    }

    @Override
    public void ciReference(String reference) {
        valueAsString(reference);
    }

    @Override
    public void ciReferences(Collection<String> references) {
        valuesAsStrings(references);
    }

    @Override
    public void typedCiReference(ConfigurationItemId ci) {
        json.writeStartObject();
        json.write("ci", ci.getId());
        if (ci.getType() != null) {
            json.write("type", ci.getType().toString());
        }
        json.writeEnd();
    }

    @Override
    public void typedCiReferences(Collection<? extends ConfigurationItemId> references) {
        json.writeStartArray();
        for (ConfigurationItemId item : references) {
            typedCiReference(item);
        }
        json.writeEnd();
    }

    @Override
    public void validationMessages(List<ValidationMessage> validations) {
        json.writeKey("validation-messages");
        json.writeStartArray();
        for (ValidationMessage validation : validations) {
            json.writeStartObject();
            json.write("level", validation.getLevel().name());
            json.write("ci", validation.getCiId());

            if (validation.getPropertyName() != null) {
                json.write("property", validation.getPropertyName());
            }
            json.write("message", validation.getMessage());
            json.writeEnd();
        }
        json.writeEnd();
    }

    @Override
    public void close() throws IOException {
        json.close();
    }

    @Override
    public void externalProperties(Map<String, ExternalProperty> externalProperties) {
        json.writeKey("external-properties");
        json.writeStartObject();
        for (Map.Entry<String, ExternalProperty> entry : externalProperties.entrySet()) {
            json.writeKey(entry.getKey());
            json.writeStartObject();
            ExternalProperty externalProperty = entry.getValue();
            if (externalProperty instanceof LookupValueKey lookupValueKey) {
                json.write("kind", "lookup");
                json.write("key", lookupValueKey.getKey());
                json.write("provider", lookupValueKey.getProviderId());
            }
            json.writeEnd();
        }
        json.writeEnd();
    }
}
