package com.xebialabs.xlrelease.variable

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.xlrelease.builder.VariableBuilder._
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType._
import com.xebialabs.xlrelease.domain.variables.reference.{PropertyUsagePoint, VariableReference}
import com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax

import java.util.{List => JList, Map => JMap, Set => JSet}
import scala.jdk.CollectionConverters._

sealed trait VariableReferenceContext

object EmptyVariableReferenceContext extends VariableReferenceContext

case class ReleaseVariableReferenceContext(release: Release) extends VariableReferenceContext

object VariableFactory {

  def createVariableByValueType(key: String, value: AnyRef, isPassword: Boolean, isGlobalOrFolder: Boolean): Variable = {
    val variableBuilder = value match {
      case v: java.lang.Boolean if !isPassword => newBooleanVariable(key, v)
      case v: java.lang.Integer if !isPassword => newIntegerVariable(key, v)
      case v: java.util.Date if !isPassword => newDateVariable(key, v)
      case v: JList[_] if !isPassword => newListStringVariable(key, v.asScala.map(e => String.valueOf(e)).asJava)
      case v: JSet[_] if !isPassword => newSetStringVariable(key, v.asScala.map(e => String.valueOf(e)).asJava)
      case v: JMap[_, _] if !isPassword => newMapStringStringVariable(key, v.asScala.map(e => String.valueOf(e._1) -> String.valueOf(e._2)).asJava)
      case v: Seq[_] if !isPassword => newListStringVariable(key, v.map(e => String.valueOf(e)).asJava)
      case v: Set[_] if !isPassword => newSetStringVariable(key, v.map(e => String.valueOf(e)).asJava)
      case v: Map[_, _] if !isPassword => newMapStringStringVariable(key, v.map(e => String.valueOf(e._1) -> String.valueOf(e._2)).asJava)
      case v: String if !isPassword => newStringVariable(key, v)
      case v: String if isPassword => newPasswordStringVariable(key, v)
      case null if isPassword => newPasswordStringVariable(key, null)
      case _ => throw new IllegalArgumentException(s"Creating ${if (isPassword) "" else "non-"}password " +
        s"variables of type ${value.getClass.getSimpleName} is not supported")
    }
    if (isGlobalOrFolder) {
      variableBuilder.withRequiresValue(false)
      variableBuilder.withShowOnReleaseStart(false)
    }
    variableBuilder.build()
  }

  def createVariableByReference(variableReference: VariableReference, context: VariableReferenceContext = EmptyVariableReferenceContext): Option[Variable] = {

    (variableReference.getType match {
      case DEFAULT => Some(classOf[StringVariable])
      case CI_PROPERTY => None
      case GLOBAL => None
      case FOLDER => None
      case SCRIPT_RESULT => Some(classOf[StringVariable])
      case PASSWORD => Some(classOf[PasswordStringVariable])
      case unknown => throw new IllegalArgumentException(s"Unexpected type [$unknown] specified for " +
        s"variable by key [${variableReference.getKey}]")

    }).map { variableClass =>
      val key = withoutVariableSyntax(variableReference.getKey)
      val variableCi = Type.valueOf(variableClass).getDescriptor
        .newInstance[Variable](null)
      variableCi.setKey(key)

      // try to guess the referenced property of the variable, this might fail :D
      if (!variableReference.getUsagePoints.isEmpty) {
        val maybeCiProperty = variableReference.getUsagePoints.asScala.find(point => point.isInstanceOf[PropertyUsagePoint]);
        maybeCiProperty.map(propertyUsagePoint => {
          val targetProperty = propertyUsagePoint.asInstanceOf[PropertyUsagePoint].getTargetProperty
          val descriptor = targetProperty.getDescriptor
          variableCi.setLabel(descriptor.getLabel)
          variableCi.setDescription(descriptor.getDescription)
        })
      }

      context match {
        case ReleaseVariableReferenceContext(release) =>
          if (release.isWorkflow) {
            //newly created variables for workflows are not required
            variableCi.setRequiresValue(false)
            variableCi.setShowOnReleaseStart(false)
          }
        case EmptyVariableReferenceContext => ()
      }

      if (variableReference.getType == SCRIPT_RESULT) {
        variableCi.setRequiresValue(false)
        variableCi.setShowOnReleaseStart(false)
      }
      variableCi
    }
  }

  def createVariableByPropertyDescriptor(propertyDescriptor: PropertyDescriptor): Option[Variable] = {
    (propertyDescriptor.getKind match {
      case PropertyKind.LIST_OF_STRING => Some(classOf[ListStringVariable])
      case PropertyKind.SET_OF_STRING => Some(classOf[SetStringVariable])
      case PropertyKind.MAP_STRING_STRING => Some(classOf[MapStringStringVariable])
      case PropertyKind.STRING if propertyDescriptor.isPassword => Some(classOf[PasswordStringVariable])
      case PropertyKind.STRING => Some(classOf[StringVariable])
      case PropertyKind.INTEGER => Some(classOf[IntegerVariable])
      case PropertyKind.BOOLEAN => Some(classOf[BooleanVariable])
      case PropertyKind.DATE => Some(classOf[DateVariable])
      case PropertyKind.CI => Some(classOf[ReferenceVariable])
      case _ => None

    }).map { variableClass =>
      Type.valueOf(variableClass.asInstanceOf[Class[_]]).getDescriptor.newInstance[Variable](null)
    }
  }

  def variableReferencesToVariables(variables: JList[VariableReference], context: VariableReferenceContext = EmptyVariableReferenceContext): JList[Variable] =
    variables.asScala.flatMap(v => VariableFactory.createVariableByReference(v, context)).asJava

}
