package com.xebialabs.xlrelease.plugins.dashboard.domain;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.springframework.util.StringUtils;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlrelease.api.internal.InternalMetadata;
import com.xebialabs.xlrelease.domain.CiWithInternalMetadata;
import com.xebialabs.xlrelease.domain.ReleaseExtension;
import com.xebialabs.xlrelease.domain.ReleaseVisitor;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Ids;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.util.Arrays.asList;

/**
 * A dashboard with tiles displayed in XL Release UI.
 */
@Metadata(versioned = false, description = "A dashboard with tiles displayed in XL Release UI")
public class Dashboard extends ReleaseExtension implements CiWithInternalMetadata {
    public static final String GLOBAL_PARENT_ID = null;

    public static final String DASHBOARD_PREFIX = "summary";

    public static final String HOME_DASHBOARD_ID = "Applications/DashboardHome";

    /**
     * An additional derived Release metadata that is used in our UI. Does not
     * belong to public API and can be changed. See implementations of com.xebialabs.xlrelease.api.internal.InternalMetadataDecorator
     * for usages.
     */
    private Map<String, InternalMetadata> $metadata = new LinkedHashMap<>();

    @Property(description = "Symbolic name for the dashboard. Max length of 255 characters.", required = false, defaultValue = "Dashboard")
    private String title;

    @Property(description = "Description for the dashboard.", required = false, size = Property.Size.MEDIUM)
    private String description;

    @Property(required = false, isTransient = true, description = "Parent that the dashboard belongs to")
    private String parentId;

    @Property(required = false, description = "The owner of this dashboard. This user has implicit read and edit permission on the dashboard")
    private String owner;

    @Property(asContainment = true, description = "Tiles in this dashboard", required = false)
    private List<Tile> tiles = new ArrayList<>();

    // Security assignments are persisted in the security subsystem
    @Property(required = false, isTransient = true, description = "List of roles or teams with view permission on this dashboard")
    private List<String> roleViewers = new ArrayList<>();

    @Property(required = false, isTransient = true, description = "List of roles or teams with edit permission on this dashboard")
    private List<String> roleEditors = new ArrayList<>();

    @Property(required = false, description = "Set number of rows the dashboard can contain")
    private Integer rows = 20;

    @Property(required = false, description = "Number of columns the dashboard can contain")
    private Integer columns = 3;

    @Property(required = false, hidden = true, defaultValue = "folder,global", description = "Determines on which levels the dashboard template can be used.")
    private List<String> supportedScopes = new ArrayList<>(asList(TileScope.FOLDER.getValue(), TileScope.GLOBAL.getValue()));

    @Property(required = false, hidden = true, description = "URI of the HTML file to render the configuration mode of this dashboard")
    private String configurationUri;

    @Property(required = false, hidden = true, description = "Custom classpath location of the dashboard configuration Jython script")
    private String scriptLocation;

    @Property(required = false, hidden = true, description = "Custom classpath location of the dashboard template JSON file")
    private String templateLocation;

    @Property(required = false, description = "Auto refresh dashboard", defaultValue = "false")
    private boolean autoRefresh;

    @Property(required = false, description = "Dashboard auto-refresh interval in seconds", defaultValue = "900")
    private Integer autoRefreshInterval = 15 * 60;

    @Property(required = false, description = "The maximum number of tiles that the dashboard can show", defaultValue = "99")
    private Integer maxTiles = 99;

    @Override
    public void setId(String id) {
        this.id = id;
        if (id != null && !this.id.equals(id)) {
            tiles.forEach(tile ->
                    tile.setId(id + Ids.SEPARATOR + Ids.getName(tile.getId()))
            );
        }
    }

    public boolean isSupportedOn(TileScope scope) {
        return supportedScopes.contains(scope.getValue());
    }

    public boolean hasOwner() {
        return owner != null && !owner.isEmpty();
    }

    public Tile getTile(String tileId) {
        return tiles.stream().filter(tile -> tileId.equals(tile.getId())).findFirst().orElseThrow(() -> new LogFriendlyNotFoundException("Tile [%s] not found", tileId));
    }

    public boolean hasConfigurationScript() {
        return StringUtils.hasText(scriptLocation);
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(final String title) {
        this.title = title;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(final String description) {
        this.description = description;
    }

    public String getParentId() {
        return parentId;
    }

    public void setParentId(final String parentId) {
        this.parentId = parentId;
    }

    public String getOwner() {
        return owner;
    }

    public void setOwner(final String owner) {
        this.owner = owner;
    }

    public List<Tile> getTiles() {
        return tiles;
    }

    public void setTiles(final List<Tile> tiles) {
        if (tiles != null) {
            checkArgument(tiles.size() <= maxTiles, String.format("The maximum number of tiles allowed for this dashboard is %s", maxTiles));
        }
        this.tiles = tiles;
    }

    public List<String> getRoleViewers() {
        return roleViewers;
    }

    public void setRoleViewers(final List<String> roleViewers) {
        this.roleViewers = roleViewers;
    }

    public List<String> getRoleEditors() {
        return roleEditors;
    }

    public void setRoleEditors(final List<String> roleEditors) {
        this.roleEditors = roleEditors;
    }

    @Override
    public void accept(ReleaseVisitor visitor) {
        super.accept(visitor);
        tiles.forEach(tile -> tile.accept(visitor));
    }

    @Override
    public Map<String, InternalMetadata> get$metadata() {
        return $metadata;
    }

    public boolean isReleaseDashboard() {
        return this.getId() != null && Ids.isReleaseId(Ids.getParentId(this.getId()));
    }

    public boolean isFolderDashboard() {
        return this.getParentId() != null && Ids.isFolderId(this.getParentId());
    }

    public boolean isGlobalDashboard() {
        return this.getParentId() == null && !isHomeDashboard();
    }

    public boolean isHomeDashboard() {
        return this.getId() != null && this.getId().equals(HOME_DASHBOARD_ID);
    }

    public Integer getRows() {
        return rows;
    }

    public Integer getColumns() {
        return columns;
    }

    public void setColumns(final Integer columns) {
        this.columns = columns;
    }

    public String getConfigurationUri() {
        return configurationUri;
    }

    public void setConfigurationUri(final String configurationUri) {
        this.configurationUri = configurationUri;
    }

    public String getScriptLocation() {
        return scriptLocation;
    }

    public void setScriptLocation(final String scriptLocation) {
        this.scriptLocation = scriptLocation;
    }

    public String getTemplateLocation() {
        return templateLocation;
    }

    public void setTemplateLocation(final String templateLocation) {
        this.templateLocation = templateLocation;
    }

    public boolean getAutoRefresh() {
        return autoRefresh;
    }

    public void setAutoRefresh(boolean autoRefresh) {
        this.autoRefresh = autoRefresh;
    }

    public Integer getAutoRefreshInterval() {
        return autoRefreshInterval;
    }

    public void setAutoRefreshInterval(Integer autoRefreshInterval) {
        this.autoRefreshInterval = autoRefreshInterval;
    }

    public Integer getMaxTiles() {
        return maxTiles;
    }

    public void setMaxTiles(Integer maxTiles) {
        this.maxTiles = maxTiles;
    }
}
