package com.xebialabs.xlrelease.domain.facet;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.Changes;
import com.xebialabs.xlrelease.domain.ReleaseVisitor;
import com.xebialabs.xlrelease.domain.VisitableItem;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;

import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;
import static java.util.stream.Collectors.toList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(root = APPLICATIONS, versioned = false, virtual = true, description = "Base facet class used to augment other CIs (tasks only currently)")
public abstract class Facet extends BaseConfigurationItem implements VisitableItem {
    @Property(hidden = true, category = "internal", description = "CI kind to which the facet can be applied.", defaultValue = "TASK")
    protected FacetScope scope = FacetScope.TASK;

    @Property(description = "The ID of the CI that the facet is applied to.")
    private String targetId;

    @Property()
    private Map<String, String> variableMapping = newHashMap();

    public FacetScope getScope() {
        return scope;
    }

    public void setScope(final FacetScope scope) {
        this.scope = scope;
    }

    @PublicApiMember
    public String getTargetId() {
        return targetId;
    }

    /**
     * @param targetId The ID of the CI that generated the record, e.g. the <code>task.id</code>
     */
    @PublicApiMember
    public void setTargetId(final String targetId) {
        this.targetId = targetId;
    }

    @Override
    public void accept(ReleaseVisitor visitor) {
        visitor.visit(this);
    }

    public List<UsagePoint> getVariableUsages() {
        return getPropertiesWithVariables().stream()
                .map(pd -> new PropertyUsagePoint(this, pd.getName()))
                .collect(toList());
    }

    public Map<String, String> getVariableMapping() {
        return variableMapping;
    }

    public void setVariableMapping(Map<String, String> variableMapping) {
        this.variableMapping = variableMapping;
    }

    public Set<String> freezeVariables(Map<String, ValueWithInterpolation> variables, Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = newHashSet();
        Collection<PropertyDescriptor> propertiesWithVariables = getPropertiesWithVariables();

        propertiesWithVariables.forEach(propertyDescriptor -> {
            Object value = getProperty(propertyDescriptor.getName());
            value = replaceAllWithInterpolation(value, variables, unresolvedVariables, freezeEvenIfUnresolved);
            setProperty(propertyDescriptor.getName(), value);
        });

        if (!propertiesWithVariables.isEmpty()) {
            changes.update(this);
        }

        return unresolvedVariables;
    }

    public Collection<PropertyDescriptor> getPropertiesWithVariables() {
        return this.getType().getDescriptor().getPropertyDescriptors()
                .stream()
                .filter(pd -> CATEGORY_INPUT.equals(pd.getCategory()))
                .collect(toList());
    }
}
