package com.xebialabs.xlrelease.variable

import com.fasterxml.jackson.databind.ObjectMapper
import com.google.common.base.Preconditions
import com.google.common.base.Preconditions.checkNotNull
import com.google.common.base.Strings._
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.{DEFAULT, FOLDER, GLOBAL, PASSWORD}
import com.xebialabs.xlrelease.utils.DateVariableUtils.printDate
import com.xebialabs.xlrelease.variable.VariableHelper.{VARIABLE_NAME_PATTERN, collectVariables, collectVariablesFromValue, containsVariables, getAllReleaseVariablesByKeys, getVariableValuesAsStrings, withVariableSyntax}

import java.lang.String.valueOf
import java.util
import java.util.regex.{Matcher, Pattern}
import java.util.{Collections, Date, Collection => JCollection, List => JList, Map => JMap, Set => JSet}
import scala.collection.mutable
import scala.collection.mutable.{Set => MSet}
import scala.jdk.CollectionConverters._

object VariableHelper extends VariableReplacementHelper with VariableCollector {

  private val VARIABLE_PATTERN: String = "\\$\\{\\s*([^}]*[^} ])\\s*\\}"
  val VARIABLE_NAME_PATTERN: Pattern = Pattern.compile(VARIABLE_PATTERN)
  private val ONLY_ONE_VARIABLE_NAME_PATTERN: Pattern = Pattern.compile("^\\$\\{([^}]+)\\}$")
  private val CI_PROPERTY_VARIABLE_NAME_PATTERN = Pattern.compile("^release\\..*$", Pattern.CASE_INSENSITIVE)
  private val GLOBAL_VARIABLE_NAME_PATTERN = Pattern.compile("^global\\..*$", Pattern.CASE_INSENSITIVE)
  private val FOLDER_VARIABLE_NAME_PATTERN = Pattern.compile("^folder\\..*$", Pattern.CASE_INSENSITIVE)

  def safeReplace(input: String, key: String, replacement: String): String = {
    if (null != input) input.replace(key, replacement) else input
  }

  /**
   * Shows if a variable looks like a ${release.something}. Note that it will return "true" also for names like ${release.custom} for which
   * {@code com.xebialabs.xlrelease.domain.variables.reference.ReleasePropertyVariableKey.isReleasePropertyVariableKey()} would return "false".
   * We discourage using such variables but they may exist in older installations.
   */
  def isCiPropertyVariable(variableName: String): Boolean =
    CI_PROPERTY_VARIABLE_NAME_PATTERN.matcher(withoutVariableSyntax(variableName)).find

  def isGlobalVariable(variableName: String): Boolean =
    GLOBAL_VARIABLE_NAME_PATTERN.matcher(withoutVariableSyntax(variableName)).find

  def isFolderVariable(variableName: String): Boolean =
    FOLDER_VARIABLE_NAME_PATTERN.matcher(withoutVariableSyntax(variableName)).find

  def isGlobalOrFolderVariable(variableName: String): Boolean = isGlobalVariable(variableName) || isFolderVariable(variableName)

  def containsVariables(input: String): Boolean = input != null && VARIABLE_NAME_PATTERN.matcher(input).find

  def formatVariableIfNeeded(variableName: String): String = {
    if (!isNullOrEmpty(variableName) && !containsOnlyVariable(variableName)) {
      withVariableSyntax(variableName)
    } else {
      variableName
    }
  }

  def containsOnlyVariable(input: String): Boolean =
    !isNullOrEmpty(input) && ONLY_ONE_VARIABLE_NAME_PATTERN.matcher(input).find

  def withVariableSyntax(variableName: String): String = "${" + variableName + "}"

  def withoutVariableSyntax(variableKey: String): String = {
    if (containsOnlyVariable(variableKey)) {
      val matcher = ONLY_ONE_VARIABLE_NAME_PATTERN.matcher(variableKey)
      matcher.find()
      matcher.group(1)
    } else {
      variableKey
    }
  }

  def checkVariable(variable: Variable): Unit = {
    checkNotNull(variable)
    variable.checkValidity()
  }

  def checkVariables(variables: JList[Variable]): Unit = {
    checkNotNull(variables)
    variables.forEach(checkVariable)
    val keys = variables.asScala.map(_.getKey)
    val uniqueKeys = MSet.empty[String]
    keys.foreach { key =>
      checkNotNull(key)
      Preconditions.checkArgument(uniqueKeys.add(key), "The variables list contains duplicate keys: '%s'", key)
    }
  }

  def getExternalVariables(variables: JList[Variable]): JMap[String, PasswordStringVariable] = {
    variables.asScala
      .filter(_.isInstanceOf[PasswordStringVariable])
      .map(v => (v.getKey, v.asInstanceOf[PasswordStringVariable]))
      .filter(e => e._2.getExternalVariableValue != null)
      .toMap
  }.asJava

  def indexByKey(variables: JList[Variable]): JMap[String, Variable] = {
    val m = new mutable.LinkedHashMap[String, Variable]()
    Option(variables).foreach(_.forEach { v => m.put(v.getKey, v) })
    m
  }.asJava

  def checkVariableIdsAreTheSame(requestVariableId: String, bodyVariableId: String): Unit = {
    Checks.checkArgument(bodyVariableId == requestVariableId,
      "Request to update variable [%s] contains an object with different ID: [%s]", requestVariableId, bodyVariableId)
  }

  def isGlobalVariableId(id: String): Boolean =
    id.startsWith("Configuration/variables/global/") || id.startsWith("/Configuration/variables/global/")

  def fillVariableValues(target: JList[Variable], source: JList[Variable]): JList[Variable] = {
    val sourceVariables: JMap[String, Variable] = indexByKey(source)
    target.forEach { targetVariable =>
      val key: String = targetVariable.getKey
      if (sourceVariables.containsKey(key)) {
        val sourceVariable: Variable = sourceVariables.get(key)
        if (!sourceVariable.isInherited) {
          if (targetVariable.getType.equals(sourceVariable.getType)) {
            targetVariable.setUntypedValue(sourceVariable.getValue)
          } else {
            throw new IllegalArgumentException(s"Cannot set value of type [${sourceVariable.getType}] into " +
              s"variable ${withVariableSyntax(sourceVariable.getKey)} of type [${targetVariable.getType}]")
          }
        }
      }
    }
    target
  }

  def cloneVariable(variable: Variable, newKey: String): Variable = {
    val newVar = VariableFactory.createVariableByValueType(newKey, variable.getValue, isPassword = false, isGlobalOrFolder = false)
    newVar.setId(null)
    newVar.setDescription(variable.getDescription)
    newVar.setLabel(variable.getLabel)
    newVar.setValueProvider(variable.getValueProvider)
    Option(newVar.getValueProvider).foreach(_.setId(null))
    newVar
  }

  def filterOutBlankStringVariables(variables: JMap[String, String]): JMap[String, String] =
    variables.asScala.filter(entry => isNotBlank(entry._2)).toMap.asJava

  def filterOutBlankValues(variables: JMap[String, ValueWithInterpolation]): JMap[String, ValueWithInterpolation] =
    variables.asScala.filter(entry => isNotBlank(entry._2.value)).toMap.asJava

  def getAllReleaseVariablesByKeys(release: Release): util.HashMap[String, Variable] = {
    val releaseVariables = release.getVariablesByKeys
    val globalVariables = if (release.getGlobalVariables != null) release.getGlobalVariables.getVariablesByKeys else Collections.emptyMap[String, Variable]
    val folderVariables = if (release.getFolderVariables != null) release.getFolderVariables.getVariablesByKeys else Collections.emptyMap[String, Variable]
    val scope = new util.HashMap[String, Variable]
    scope.putAll(releaseVariables)
    scope.putAll(globalVariables)
    scope.putAll(folderVariables)
    scope
  }

  private lazy val jsonWriter = new ObjectMapper()

  def toString(untyped: Any): String = untyped match {
    case value: String => value
    case value: Date => printDate(value)
    case value: util.Map[Any@unchecked, Any@unchecked] => jsonWriter.writeValueAsString(value)
    case value: util.Collection[Any@unchecked] => jsonWriter.writeValueAsString(value)
    case _ => valueOf(untyped)
  }
}

trait VariableReplacementHelper {
  def replaceAll(raw: String, replacements: JMap[String, String]): String = {
    // replacements should be "with variable syntax" ie surrounded with '${}' otherwise output may not be what you expect
    Option(raw).map(replacements.entrySet().asScala.foldLeft(_) { (s, entry) =>
      val matcher = VARIABLE_NAME_PATTERN.matcher(entry.getKey)
      if (matcher.matches() && entry.getValue != null) {
        // replaceAll is used in order go ignore spaces around variable name (like '${testVar }')
        // it has two tricks:
        // 1. we need to quote variable name since we allow any characters as variable name
        // 2. we need to "escape" dollar signs and backslashes since we need it to be literal replacement (which quoteReplacement essentially does for us)
        s.replaceAll("\\$\\{\\s*" + Pattern.quote(matcher.group(1)) + "\\s*\\}", Matcher.quoteReplacement(entry.getValue))
      } else {
        if (entry.getValue != null) {
          s.replace(entry.getKey, entry.getValue)
        } else {
          s
        }
      }
    }).orNull
  }

  def replaceAll[T](raw: T, replacements: JMap[String, String], unresolvedVariables: JSet[String], freezeEvenIfUnresolved: Boolean): T = {
    val expandedReplacements = replacements.asScala.map(rep => rep._1 -> ValueWithInterpolation(rep._2, preventInterpolation = false)).asJava
    replaceAllWithInterpolation(raw, expandedReplacements, unresolvedVariables, freezeEvenIfUnresolved)
  }

  private def replaceAllWithInterpolation(raw: String, replacements: JMap[String, ValueWithInterpolation], unresolvedVariables: JSet[String],
                                          freezeEvenIfUnresolved: Boolean): String = {
    val replacementsValues = replacements.asScala.values.filter(_.preventInterpolation).map(_.value).toSeq.asJava
    val varsFromNonInterpolatableReplacements = collectVariables(replacementsValues)

    val filteredReplacements = replacements.asScala
      .filter(rep => rep._2.value != null && !varsFromNonInterpolatableReplacements.contains(rep._1))
      .map { case (key, value) => (key, value.value) }
      .asJava

    Option(replaceAll(raw, filteredReplacements)).map { replaced =>
      val remainingVariables = collectVariables(replaced)
      val filteredRemaining = remainingVariables.asScala.filter(v => !varsFromNonInterpolatableReplacements.contains(v)).asJava

      unresolvedVariables.addAll(filteredRemaining)
      if (freezeEvenIfUnresolved) {
        filteredRemaining.asScala.foldLeft(replaced) { (s, remainingVariable) =>
          freezeUnresolvedVariable(s, remainingVariable)
        }
      } else {
        replaced
      }
    }.orNull
  }

  /*
   * It will replace all placeholders in raw string (1st param), with values from replacements map, matched by placeholder+map key pair. However, before the
   * replacements are made they will be filtered for enries whose key is amongst non-interpolable replacements.
   *
   * E.g. if you have a raw string equal to "${car} ${nonInter}" and a nonInter variable with "some value ${car}", the ${car} will not be replaced.
   *
   * As a side effect this method will add unresolvable placeholders from raw string to unresolvedVariables
   *
   * @return raw string with all replacements made. Can be null.
   */
  def replaceAllWithInterpolation[T](raw: T, replacements: JMap[String, ValueWithInterpolation], unresolvedVariables: JSet[String],
                                     freezeEvenIfUnresolved: Boolean): T = {
    def replaceInValue(value: String) = replaceAllWithInterpolation(value, replacements, unresolvedVariables, freezeEvenIfUnresolved)

    Option(raw).map {
      case v: String => replaceInValue(v)
      case v: JList[String@unchecked] => v.asScala.map(replaceInValue).asJava
      case v: JSet[String@unchecked] => v.asScala.map(replaceInValue).asJava
      case v: JMap[String@unchecked, String@unchecked] => v.asScala.map(t => replaceInValue(t._1) -> replaceInValue(t._2)).asJava
      case v => v
    }.orNull.asInstanceOf[T]
  }

  def getVariableValuesAsStrings(vars: JList[Variable]): JMap[String, String] =
    getVariableValuesAsStrings(vars.asScala.toList, passwords = false)

  def getPasswordVariableValuesAsStrings(vars: JList[Variable]): JMap[String, String] =
    getVariableValuesAsStrings(vars.asScala.toList, passwords = true)

  private def getVariableValuesAsStrings(variables: List[Variable], passwords: Boolean): JMap[String, String] = mutable.HashMap.newBuilder.addAll({
    val interpolatableVars = resolveInterpolatableVariables(variables, passwords)
    val nonInterpolatableVars = variables.filter(v => v.isInstanceOf[StringVariable] && v.asInstanceOf[StringVariable].isPreventInterpolation)
      .map(v => withVariableSyntax(v.getKey) -> v.getValueAsString).toMap

    val fullyResolvedInterpolatableVars = interpolatableVars.filter(entry => collectVariablesFromValue(entry._2).isEmpty)
    val updatedInterpolatable = interpolatableVars.filter(entry => collectVariablesFromValue(entry._2).nonEmpty)
      .map { case (key, value) => key -> replaceAll(value, nonInterpolatableVars.asJava)}

    updatedInterpolatable ++ fullyResolvedInterpolatableVars ++ nonInterpolatableVars
  }).result().asJava

  private def resolveInterpolatableVariables(variables: List[Variable], passwords: Boolean) = {
    val nonInterpolableVars = variables.filter(v => v.isPassword == passwords
      && (!v.isInstanceOf[StringVariable] || !v.asInstanceOf[StringVariable].isPreventInterpolation))
    VariablesGraphResolver.resolveAsStrings(nonInterpolableVars.toSet)._1.toSeq.toMap
  }

  def freezeUnresolvedVariable(input: String, variableName: String): String = {
    if (containsVariables(variableName)) {
      val dollarLessName: String = variableName.substring(1)
      input.replace(variableName, "$~" + dollarLessName)
    } else {
      input
    }
  }
}

trait VariableCollector {

  def collectVariables(input: AnyRef): JSet[String] = {
    Option(input).map {
      case value: String => collectVariablesFromValue(value)
      case value: JCollection[_] => value.asScala.flatMap(collectVariablesFromValue).toSet
      case value: JMap[_, _] => value.asScala.flatMap { case (k, v) => collectVariablesFromValue(k) ++ collectVariablesFromValue(v) }.toSet
      case _ => Set.empty[String]
    }.getOrElse(Set.empty[String]).asJava
  }

  def collectVariablesFromValue(value: Any): Set[String] = {
    val result = MSet.empty[String]
    value match {
      case v: String =>
        // ignore spaces around variable name like '${testVar }'
        val matcher = VARIABLE_NAME_PATTERN.matcher(v)
        while (matcher.find) {
          result.add(s"$${${matcher.group(1)}}")
        }
      case _ =>
    }
    result.toSet
  }

  def getUsedExternalPasswordVariables(release: Release): JMap[String, PasswordStringVariable] = {
    getUsedVariables(release, enablePassword = true)
      .asScala
      .filter {
        case passwordVariable: PasswordStringVariable => passwordVariable.getExternalVariableValue != null
        case _ => false
      }
      .map { v => (v.getKey, v.asInstanceOf[PasswordStringVariable]) }
      .toMap
      .asJava
  }

  def getUsedStringVariables(release: Release): JMap[String, String] = {
    val variableList = getUsedVariables(release)
    getVariableValuesAsStrings(variableList)
  }

  private def getUsedVariables(release: Release, enablePassword: Boolean = false) = {
    val allUsedVariablesMap = new util.HashMap[String, Variable]
    val scope: util.Map[String, Variable] = getAllReleaseVariablesByKeys(release)
    for (variableReference <- release.collectVariableReferences.asScala) {
      val variableKey = VariableHelper.withoutVariableSyntax(variableReference.getKey)
      val variable = scope.get(variableKey)
      if (variable != null) {
        variableReference.getType match {
          case DEFAULT =>
            allUsedVariablesMap.putIfAbsent(variableKey, variable)
          case GLOBAL =>
            collectVariablesInScope(allUsedVariablesMap, scope, variableKey)
          case FOLDER =>
            collectVariablesInScope(allUsedVariablesMap, scope, variableKey)
          case PASSWORD => if (enablePassword) {
            collectVariablesInScope(allUsedVariablesMap, scope, variableKey)
          }
          case _ => // nothing
        }
      }
    }
    val allUsedVariablesList = new util.ArrayList[Variable](allUsedVariablesMap.values)
    allUsedVariablesList.addAll(release.getCiPropertyVariables)
    allUsedVariablesList
  }

  def collectVariablesInScope(allUsedVariablesMap: JMap[String, Variable], scope: JMap[String, Variable], variableKeyToResolve: String): Unit = {
    if (!allUsedVariablesMap.containsKey(variableKeyToResolve)) {
      val variableToResolve: Variable = scope.get(variableKeyToResolve)
      if (null != variableToResolve) {
        allUsedVariablesMap.putIfAbsent(variableKeyToResolve, variableToResolve)
        var referencedVariableKeys: mutable.Set[String] = collectVariables(variableToResolve.getValueAsString).asScala
        referencedVariableKeys = referencedVariableKeys.map(VariableHelper.withoutVariableSyntax)
        for (referencedVariableKey <- referencedVariableKeys) {
          if (!allUsedVariablesMap.containsKey(referencedVariableKey)) {
            val referencedVariable: Variable = scope.get(referencedVariableKey)
            if (referencedVariable != null) {
              collectVariablesInScope(allUsedVariablesMap, scope, referencedVariableKey)
              allUsedVariablesMap.put(referencedVariableKey, referencedVariable)
            }
          }
        }
      }
    }
  }
}