package com.xebialabs.xlrelease.plugins.dashboard.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlrelease.domain.ReleaseExtension;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

/**
 * Base class for dashboard tiles, for example tiles of the "Release contents" page.
 */
@Metadata(virtual = true, versioned = false, description = "Base type for dashboard tiles")
public class Tile extends ReleaseExtension {
    public static final String PROPERTY_USER_SPECIFIC_CACHE = "userSpecificCache";
    public static final String PROPERTY_EXPIRATION_TIME = "expirationTime";
    public static final String PROPERTY_MAX_CACHE_ENTRIES = "maxCacheEntries";

    @Property(category = CATEGORY_INPUT, description = "Title of this tile")
    private String title;

    @Property(required = false, hidden = true, size = Property.Size.MEDIUM, description = "Describes the functionality of this tile.")
    private String description;

    @Property(required = true, hidden = true, description = "URI of the HTML file to render the summary mode of this tile")
    private String uri;

    @Property(required = false, hidden = true, description = "URI pointing to the HTML file to render when we are switching to details view")
    private String detailsUri;

    @Property(required = false, hidden = true, description = "URI of the HTML file to render the configuration mode of this tile")
    private String configurationUri;

    @Property(required = false, hidden = true, description = "Dashboards that contain this tile by default")
    private List<String> defaultDashboards = new ArrayList<>();

    @Property(required = false, hidden = true, description = "Optional custom classpath location of the tile Jython script")
    private String scriptLocation;

    @Property(required = false, hidden = true, description = "Optional custom classpath location of the Jython script for configuring the tile on create")
    private String provisioningScriptLocation;

    // The gridConfiguration properties get their defaults on UI side, so they have to be null by default
    @Property(category = "gridConfiguration", description = "Default width of this tile in number of dashboard columns it occupies")
    private Integer width;
    @Property(category = "gridConfiguration", description = "Default height of this tile in number of dashboard rows it occupies")
    private Integer height;
    @Property(category = "gridConfiguration", description = "Row number of this tile on the dashboard")
    private Integer row;
    @Property(category = "gridConfiguration", description = "Column number of this tile on the dashboard")
    private Integer col;
    @Property(category = "gridConfiguration", description = "Minimum height size of this tile on the dashboard")
    private Integer minSizeY;

    @Property(category = "cache", required = false, hidden = true, defaultValue = "true", description = "True if tile data should be cached")
    private boolean cacheEnabled = true;

    @Property(category = "cache", required = false, hidden = true, description = "True if tile data should be cached per user.")
    private boolean userSpecificCache = false;

    @Property(category = "cache", required = false, hidden = true, defaultValue = "300", description = "Expiration time for a tile cache (in seconds).")
    private Integer expirationTime = 5 * 60;

    @Property(category = "cache", required = false, hidden = true, defaultValue = "500", description = "Maximum cache entries.")
    private Integer maxCacheEntries = 500;

    @Property(required = false, hidden = true, defaultValue = "false", description = "Render content inside the main page without using iframe element.")
    private boolean embedded;

    @Property(required = false, hidden = true, defaultValue = "release", description = "Determines on which levels the tile can be used.")
    private List<String> supportedScopes = singletonList(TileScope.RELEASE.getValue());

    @Property()
    private Map<String, String> variableMapping = newHashMap();

    @Override
    public List<UsagePoint> getVariableUsages() {
        return getProperties()
                .stream()
                .map(pd -> new PropertyUsagePoint(this, pd.getName()))
                .collect(toList());
    }

    public List<PropertyDescriptor> getProperties() {
        return this.getType().getDescriptor().getPropertyDescriptors()
                .stream()
                .filter(pd -> CATEGORY_INPUT.equals(pd.getCategory()))
                .collect(toList());
    }

    public static Tile newDummy() {
        Tile tile = new Tile();
        tile.setId(null);
        return tile;
    }

    public boolean isSupportedOn(TileScope scope) {
        return supportedScopes.contains(scope.getValue());
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(final String title) {
        this.title = title;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(final String description) {
        this.description = description;
    }

    public String getUri() {
        return uri;
    }

    public void setUri(final String uri) {
        this.uri = uri;
    }

    public String getDetailsUri() {
        return detailsUri;
    }

    public void setDetailsUri(final String detailsUri) {
        this.detailsUri = detailsUri;
    }

    public String getConfigurationUri() {
        return configurationUri;
    }

    public void setConfigurationUri(final String configurationUri) {
        this.configurationUri = configurationUri;
    }

    public List<String> getDefaultDashboards() {
        return defaultDashboards;
    }

    public Integer getWidth() {
        return width;
    }

    public void setWidth(final Integer width) {
        this.width = width;
    }

    public Integer getRow() {
        return row;
    }

    public void setRow(final Integer row) {
        this.row = row;
    }

    public Integer getCol() {
        return col;
    }

    public void setCol(final Integer col) {
        this.col = col;
    }

    public Integer getMinSizeY() {
        return minSizeY;
    }

    public void setMinSizeY(Integer minSizeY) {
        this.minSizeY = minSizeY;
    }

    public Map<String, String> getVariableMapping() {
        return variableMapping;
    }

    public void setVariableMapping(Map<String, String> variableMapping) {
        this.variableMapping = variableMapping;
    }

    public boolean isCacheEnabled() {
        return cacheEnabled;
    }

    public void setCacheEnabled(final boolean cacheEnabled) {
        this.cacheEnabled = cacheEnabled;
    }

    public boolean isUserSpecificCache() {
        return userSpecificCache;
    }

    public void setUserSpecificCache(final boolean userSpecificCache) {
        this.userSpecificCache = userSpecificCache;
    }

    public Integer getExpirationTime() {
        return expirationTime;
    }

    public void setExpirationTime(final Integer expirationTime) {
        this.expirationTime = expirationTime;
    }

    public Integer getMaxCacheEntries() {
        return maxCacheEntries;
    }

    public void setMaxCacheEntries(final Integer maxCacheEntries) {
        this.maxCacheEntries = maxCacheEntries;
    }

    public String getScriptLocation() {
        return scriptLocation;
    }

    public String getProvisioningScriptLocation() {
        return provisioningScriptLocation;
    }

    public boolean isEmbedded() {
        return embedded;
    }

    public void setEmbedded(final boolean embedded) {
        this.embedded = embedded;
    }

    public List<String> getSupportedScopes() {
        return supportedScopes;
    }

    public void setSupportedScopes(final List<String> supportedScopes) {
        this.supportedScopes = supportedScopes;
    }

    public Integer getHeight() {
        return height;
    }

    public void setHeight(final Integer height) {
        this.height = height;
    }
}
