package com.xebialabs.xlrelease.domain.variables.reference;

import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_OUTPUT;
import static com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.CI_PROPERTY;
import static com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.DEFAULT;
import static com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.FOLDER;
import static com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.GLOBAL;
import static com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.PASSWORD;
import static com.xebialabs.xlrelease.domain.variables.reference.VariableReference.VariableUsageType.SCRIPT_RESULT;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static com.xebialabs.xlrelease.variable.VariableHelper.containsOnlyVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isCiPropertyVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isFolderVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isGlobalVariable;
import static java.util.function.Function.identity;

/**
 * This usage point means that given property <strong>value</strong> mentions a variable,
 * e.g. a string property "Some ${var}" or a list property ['value1', 'value 2 with ${var}"].
 */
public class PropertyUsagePoint implements UsagePoint {
    private CiProperty property;
    private VariableUsageType defaultType;

    public PropertyUsagePoint(CiProperty ciProperty) {
        this(ciProperty, DEFAULT);
    }

    public PropertyUsagePoint(CiProperty ciProperty, VariableUsageType defaultType) {
        this.property = ciProperty;
        this.defaultType = defaultType;
    }

    public PropertyUsagePoint(ConfigurationItem item, String property) {
        this(CiProperty.of(item, property).get());
    }

    public PropertyUsagePoint(ConfigurationItem item, String property, VariableUsageType defaultType) {
        this(CiProperty.of(item, property).get(), defaultType);
    }

    VariableUsageType getVariableUsageType(String variableName) {
        VariableUsageType type = defaultType;
        if (isCiPropertyVariable(variableName)) {
            type = CI_PROPERTY;
        } else if (isGlobalVariable(variableName)) {
            type = GLOBAL;
        } else if (isFolderVariable(variableName)) {
            type = FOLDER;
        } else if (property.isPassword()) {
            type = PASSWORD;
        } else if (CATEGORY_OUTPUT.equals(property.getCategory())) {
            type = SCRIPT_RESULT;
        } else if (property.getKind() == STRING && !containsOnlyVariable(property.getValue())) {
            type = DEFAULT;
        }
        return type;
    }

    @Override
    public Map<String, VariableUsageType> collectVariables() {
        return VariableHelper.collectVariables(property.getValue())
                .stream()
                .collect(toMap(identity(), this::getVariableUsageType));
    }

    @Override
    public Set<ConfigurationItem> replaceVariable(Variable variable, VariableOrValue replacement) {
        property.replaceInValue(variable, replacement);
        return newHashSet(property.getParentCi());
    }

    public CiProperty getTargetProperty() {
        return property;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        final PropertyUsagePoint that = (PropertyUsagePoint) o;

        return this.property.equals(that.property);
    }

    @Override
    public int hashCode() {
        int result = property.hashCode();
        result = 31 * result + property.hashCode();
        return result;
    }
}
