package com.xebialabs.xlrelease.domain.variables;

import java.util.HashMap;
import java.util.Map;
import com.google.common.base.Joiner;

import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;

@PublicApiRef
@ShowOnlyPublicApiMembers
public class MapStringStringVariable extends Variable.VariableWithValue<Map<String, String>> {

    @PublicApiMember
    @Property(required = false, description = "Value of the release variable or default value of the template variable")
    protected Map<String, String> value;

    @Override
    public Map<String, String> getValue() {
        return value == null ? null : newHashMap(value);
    }

    @Override
    public Map<String, String> getEmptyValue() {
        return new HashMap<>();
    }

    @Override
    public boolean isValueEmpty() {
        return value == null || value.isEmpty();
    }

    @PublicApiMember
    @Override
    public void setValue(final Map<String, String> value) {
        this.value = value;
    }

    @Override
    public void setUntypedValue(Object newValue) {
        if (newValue == null) {
            this.setValue(newHashMap());
        } else if (newValue instanceof Map) {
            @SuppressWarnings("unchecked")
            Map<String, String> value = ((Map<Object, Object>) newValue).entrySet().stream()
                                .collect(
                                    toMap(
                                            k -> VariableHelper.toString(k.getKey()),
                                            v -> VariableHelper.toString(v.getValue())
                                    )
                                );
            this.setValue(value);
        } else {
            throw new IllegalArgumentException(format("Cannot set instance of %s into map of string string variable value: %s",
                    newValue.getClass().getSimpleName(), newValue));
        }
    }

    @Override
    public String getValueAsString() {
        return formatValueAsString(getValue());
    }

    @Override
    public boolean isPassword() {
        return false;
    }

    @Override
    public String getEmptyValueAsString() {
        return formatValueAsString(getEmptyValue());
    }

    @Override
    public boolean isValueAssignableFrom(final Object value) {
        return value instanceof Map;
    }

    private String formatValueAsString(final Map<String, String> valueToFormat) {
        Map<String, String> mappedValue = valueToFormat
                .entrySet()
                .stream()
                .collect(toMap(
                        k -> String.format("'%s'", escapeQuotes(k.getKey())),
                        v -> String.format("'%s'", escapeQuotes(VariableHelper.toString(v.getValue())))));

        return format("{%s}", Joiner.on(", ").withKeyValueSeparator(": ").join(mappedValue));
    }

}
