package com.xebialabs.xlrelease.domain.events

import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.domain.{CreateReleaseTask, Release}

import java.util.{List => JList}

/**
  * Common interface for domain events related to release execution or modification.
  */
sealed trait ReleaseEvent extends XLReleaseEvent

//
// Release CUD
//

/**
  * Event published when a release or a template has been created.
  * @param release the created release or template. You can detect if it is a release or a template from its status.
  * @param source the way the release was created: one of [[Imported]], [[CreatedFromDsl]], [[CreatedWithoutTemplate]], [[CreatedFromTemplate]] or [[CreatedFromCreateReleaseTask]].
  *               If this is `null` then the release was created from UI or API from scratch.
  */
case class ReleaseCreatedEvent(release: Release, source: ReleaseCreationSource) extends ReleaseEvent

sealed trait ReleaseCreationSource
case class Imported() extends ReleaseCreationSource
case class CreatedFromDsl() extends ReleaseCreationSource
case class CreatedWithoutTemplate() extends ReleaseCreationSource
case class CreatedFromTemplate(templateId: String) extends ReleaseCreationSource
case class CreatedFromCreateReleaseTask(task: CreateReleaseTask, templateId: String) extends ReleaseCreationSource
case class CreatedFromTemplateByTrigger(templateId: String, triggerId: String) extends ReleaseCreationSource
case class CreatedFromAsCode(scmTraceabilityData: Option[SCMTraceabilityData]) extends ReleaseCreationSource

/**
  * Event published when a release has been duplicated.
  * @param releaseDuplicate the duplicated instance of the release.
  */
case class ReleaseDuplicatedEvent(releaseDuplicate: Release) extends ReleaseEvent

/**
  * Event published when a template or a release has been moved to another folder.
  * @param fromReleaseId the original ID of the template or release.
  * @param toReleaseId the new ID of the template or release.
  */
case class ReleaseMovedEvent(fromReleaseId: String, toReleaseId: String) extends ReleaseEvent

/**
  * Event published when properties of a release or a template has been updated. This means the properties of a release
  * itself only, for example release title. If a title of underlying phase is updated, then [[PhaseUpdatedEvent]] is published.
  * @param original a copy of the original release or template before update.
  * @param updated the new updated release or template instance. You can use [[original]] to compare which properties have changed.
  */
case class ReleaseUpdatedEvent(original: Release, updated: Release) extends ReleaseEvent

/**
 * Event published when a template has been updated via as code.
 * @param release the template that was updated via as code.
 * @param scmTraceabilityData the scm traceability data if as code was the result of scm operation, None otherwise.
 */
case class ReleaseUpdatedFromAsCodeEvent(release: Release, scmTraceabilityData: Option[SCMTraceabilityData]) extends ReleaseEvent

/**
  * Event published when a release or a template has been deleted.
  * @param release a copy of the deleted release or template. Note that when this event is
  *                handled the release does not exist in the repository anymore.
  */
case class ReleaseDeletedEvent(release: Release) extends ReleaseEvent

/**
  * Event published when a release is overdue.
  * @param release the instance of the release.
  */
case class ReleaseOverdueEvent(release: Release) extends ReleaseEvent


/**
  * Event published when template has been restored from previous revision.
  *
  * @param original a copy of the original release or template before update.
  * @param updated the new updated release or template instance. You can use [[original]] to compare which properties have changed.
  * @param revisionId id of the revision that was restored.
  */
case class ReleaseRestoredEvent(original: Release, updated: Release, revisionId: Int) extends ReleaseEvent
//
// Release execution
//

/**
  * Event published when a release has started.
  * @param release the instance of started release. Note that it will not necessarily be in the `IN_PROGRESS` state:
  *                for example, if a release has no tasks, then this instance will already be in the `COMPLETED` state.
  * @param isPartOfBulkOperation flag indicating if this release was started as a part of bulk start release action.
  */
case class ReleaseStartedEvent(release: Release, isPartOfBulkOperation: Boolean = false) extends ReleaseExecutionEvent

/**
  * Event published when a release has been paused as part of a phase restart.
  * @param release the release instance.
  */
case class ReleasePausedEvent(release: Release) extends ReleaseExecutionEvent

/**
  * Event published when a release has been resumed after a phase restart.
  * @param release the release instance.
  */
case class ReleaseResumedEvent(release: Release) extends ReleaseExecutionEvent

/**
  * Event published when a release goes into `FAILED` state.
  * @param release the release instance.
  */
case class ReleaseFailedEvent(release: Release) extends ReleaseExecutionEvent

/**
  * Event published when a release goes into `FAILING` state. This happens when a parallel group is active and it
  * has both `IN_PROGRESS` and `FAILED` tasks.
  * @param release the release instance.
  */
case class ReleaseStartedFailingEvent(release: Release) extends ReleaseExecutionEvent

/**
  * Event published when one of release's tasks is retried. More specifically it happens when release goes
  * from `FAILED` or `FAILING` state into `IN_PROGRESS` state.
  * @param release the release instance.
  */
case class ReleaseRetriedEvent(release: Release) extends ReleaseExecutionEvent

/**
  * Event published when a release is aborted.
  * @param release the release instance.
  * @param isPartOfBulkOperation flag indicating if this release was aborted as a part of bulk abort release action.
  */
case class ReleaseAbortedEvent(release: Release, isPartOfBulkOperation: Boolean = false) extends ReleaseExecutedEvent

/**
  * Event published when a release is completed.
  * @param release the release instance.
  */
case class ReleaseCompletedEvent(release: Release) extends ReleaseExecutedEvent


/**
  * Common interface for events when the release status changes.
  */
sealed trait ReleaseExecutionEvent extends ReleaseEvent {
  /**
    * The release of which status has changed.
    */
  val release: Release
}

sealed trait ReleaseBulkEvent extends ReleaseEvent {
  val releases: JList[Release]
}

case class ReleaseBulkStartedEvent(releases: JList[Release]) extends ReleaseBulkEvent
case class ReleaseBulkAbortedEvent(releases: JList[Release]) extends ReleaseBulkEvent

/**
  * A parent event published when a release is finished: completed or aborted. So this is an
  * instance of either [[ReleaseCompletedEvent]] or [[ReleaseAbortedEvent]].
  */
sealed trait ReleaseExecutedEvent extends ReleaseExecutionEvent


/**
  * Event published when a release is started from a Create Release task.
  * @param task the Create Release task which created a release and started it.
  * @param release the created and started release.
  */
case class ReleaseStartedFromCreateReleaseTaskEvent(task: CreateReleaseTask, release: Release) extends ReleaseEvent
