package com.xebialabs.xlrelease.domain.delivery;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.delivery.conditions.ConditionGroup;
import com.xebialabs.xlrelease.domain.utils.DeliveryUtils;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Ids;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static java.util.stream.Collectors.toList;

@Metadata(versioned = false)
@PublicApiRef
@ShowOnlyPublicApiMembers
public class Transition extends BaseConfigurationItem {

    @Property
    private String title;

    @Property(asContainment = true)
    private Stage stage;

    @Property(asContainment = true, description = "Conditions for the transition to execute automatically.")
    private List<Condition> conditions = new ArrayList<>();

    @Property(label = "Is automated?", defaultValue = "true", description = "Are the automated conditions on the transition enabled")
    private boolean automated = true;

    //

    public Transition() {

    }

    public Transition(final String title) {
        this.title = title;
    }

    @PublicApiMember
    public <C extends Condition> List<C> getConditionsOfType(Class<C> conditionClass) {
        Type conditionType = Type.valueOf(conditionClass);
        return getAllConditions().stream()
                .filter(condition -> condition.getType().instanceOf(conditionType))
                .map(conditionClass::cast)
                .collect(toList());
    }

    public List<Condition> getAllConditions() {
        return getConditions().stream().flatMap(child -> child.getAllConditions().stream()).collect(toList());
    }

    public List<Condition> getLeafConditions() {
        return getConditions().stream().flatMap(child -> child.getLeafConditions().stream()).collect(toList());
    }

    public void addCondition(final Condition condition) {
        if (conditions == null) {
            conditions = new ArrayList<>();
        }
        conditions.add(condition);
    }

    public Condition getRootCondition() {
        ConditionGroup root = new ConditionGroup();
        root.setId("ROOT");
        root.setOperator(ConditionGroup.Operator.OR);
        root.setConditions(this.conditions);
        return root;
    }

    public Condition getConditionById(String conditionId) {
        return findConditionById(conditionId)
                .orElseThrow(() -> new LogFriendlyNotFoundException("Condition '%s' does not exist in transition '%s'", conditionId, title));
    }

    public Optional<Condition> findConditionById(String conditionId) {
        String conditionBaseId = Ids.getName(conditionId);
        return getAllConditions().stream().filter(condition -> Ids.getName(condition.getId()).equals(conditionBaseId)).findFirst();
    }

    public void checkRestrictions() {
        checkArgument(conditions.size() <= 1, "Conditions can contain only one root condition group");
        conditions.forEach(condition -> {
            checkArgument(DeliveryUtils.isConditionGroup(condition), "The root condition must be a condition group");
            ConditionGroup rootGroup = (ConditionGroup) condition;
            checkNotNull(rootGroup.getOperator(), "Root group operator");

            rootGroup.getConditions().forEach(nestedCondition -> {
                checkArgument(DeliveryUtils.isConditionGroup(nestedCondition), "Root group can only contain other nested condition groups");
                ConditionGroup nestedGroup = (ConditionGroup) nestedCondition;
                checkNotNull(nestedGroup.getOperator(), "Nested group operator");
                checkArgument(!rootGroup.getOperator().equals(nestedGroup.getOperator()), "Nested group operator must not equal the root group operator");

                nestedGroup.getConditions().forEach(concreteCondition -> {
                    checkArgument(!DeliveryUtils.isConditionGroup(concreteCondition), "Nested groups can only contain concrete conditions");
                });
            });
        });
    }

    //

    @PublicApiMember
    public String getTitle() {
        return title;
    }

    @PublicApiMember
    public void setTitle(final String title) {
        this.title = title;
    }

    @PublicApiMember
    public Stage getStage() {
        return stage;
    }

    @PublicApiMember
    public void setStage(Stage stage) {
        this.stage = stage;
    }

    @PublicApiMember
    public List<Condition> getConditions() {
        return conditions;
    }

    @PublicApiMember
    public void setConditions(final List<Condition> conditions) {
        this.conditions = conditions;
    }

    @PublicApiMember
    public boolean isAutomated() {
        return automated;
    }

    @PublicApiMember
    public void setAutomated(final boolean automated) {
        this.automated = automated;
    }
}
