package com.xebialabs.xlrelease.domain.delivery

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.{Metadata, Property}
import com.xebialabs.xlrelease.domain.delivery.SubscriptionResult.{completed, failed, inProgress}
import com.xebialabs.xlrelease.domain.delivery.SubscriptionStatus._
import com.xebialabs.xlrelease.domain.delivery.TrackedItemStatus.{NOT_READY, READY, SKIPPED}

import java.util
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

object SubscriptionResult {
  def completed(message: String): SubscriptionResult = SubscriptionResult(COMPLETED, message)

  def failed(message: String): SubscriptionResult = SubscriptionResult(FAILED, message)

  def inProgress(message: String = ""): SubscriptionResult = SubscriptionResult(IN_PROGRESS, message)
}

case class SubscriptionResult(@BeanProperty status: SubscriptionStatus,
                              @BeanProperty message: String)

@Metadata(versioned = false)
abstract class Subscriber extends BaseConfigurationItem {
  @BeanProperty
  @Property
  var sourceId: String = _

  def validate(delivery: Delivery): Unit

  def evaluate(delivery: Delivery): SubscriptionResult

  def isEqual(subscriber: Subscriber): Boolean = {
    sourceId == subscriber.sourceId
  }
}

class StageCompletionSubscriber extends Subscriber {

  @BeanProperty
  @Property
  var stageId: String = _

  override def validate(delivery: Delivery): Unit = {
    delivery.getStageByIdOrTitle(this.stageId)
  }

  override def evaluate(delivery: Delivery): SubscriptionResult = {
    val stage = delivery.getStageByIdOrTitle(stageId)
    if (stage.getStatus == StageStatus.CLOSED) {
      completed(s"Stage '${stage.getTitle}' is already completed\n")
    } else {
      inProgress()
    }
  }
}

case class ResolvedItem(item: TrackedItem, statusInStage: Option[TrackedItemStatus])

class TrackedItemStatusChangeSubscriber extends Subscriber {
  @BeanProperty
  @Property
  var stageId: String = _
  @BeanProperty
  @Property
  var itemIds: java.util.List[String] = new util.ArrayList[String]()
  @BeanProperty
  @Property
  var itemStatus: TrackedItemStatus = _

  override def validate(delivery: Delivery): Unit = {
    delivery.getStageByIdOrTitle(this.stageId)
    itemIds.forEach(itemId => delivery.getItemByIdOrTitle(itemId))
  }

  override def evaluate(delivery: Delivery): SubscriptionResult = {
    val stage = delivery.getStageByIdOrTitle(stageId)
    val resolvedItems = itemIds.asScala.map { itemId =>
      ResolvedItem(delivery.getItemByIdOrTitle(itemId), stage.findItemById(itemId).asScala.map(_.getStatus))
    }

    if (resolvedItems.exists(_.item.isDescoped)) {
      failed(s"Unable to complete task, the following tracked items are de-scoped:\n " +
        s"* ${resolvedItems.filter(_.item.isDescoped).map(_.item.getTitle).sorted.mkString("\n * ")}")
    } else if (itemStatus == NOT_READY && resolvedItems.forall(_.statusInStage.isDefined)) {
      completed(s"All tracked items are available in stage '${stage.getTitle}'")
    } else if (itemStatus == READY && resolvedItems.exists(_.statusInStage.exists(status => status == SKIPPED))) {
      failed(s"Unable to complete task, the following tracked items are skipped in stage '${stage.getTitle}':\n " +
        s"* ${resolvedItems.filter(_.statusInStage.exists(status => status == SKIPPED)).map(_.item.getTitle).sorted.mkString("\n * ")}")
    } else if (itemStatus == READY && resolvedItems.forall(_.statusInStage.exists(status => status == READY))) {
      completed(s"All tracked items are completed in stage '${stage.getTitle}'")
    } else {
      inProgress()
    }
  }

  def matches(delivery: Delivery, itemId: String, stageId: Option[String] = None, status: Option[TrackedItemStatus] = None): Boolean = {
    this.itemIds.contains(itemId) &&
      (stageId.isEmpty || this.stageId == stageId.get) &&
      (status.isEmpty || (this.itemStatus == status.get && allItemsHaveStatus(delivery, stageId.get, status.get)))
  }

  private def allItemsHaveStatus(delivery: Delivery, stageId: String, status: TrackedItemStatus): Boolean = {
    val stage = delivery.getStageByIdOrTitle(stageId)
    itemIds.asScala.forall {
      stage.findItemById(_).asScala.exists(stageItem => stageItem.getStatus == status)
    }
  }

}
