package com.xebialabs.xlrelease.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Script: Jython Script", versioned = false)
public class ScriptTask extends ResolvableScriptTask {

    public static final String JYTHON_ENGINE = "jython";

    @Property
    private String script;

    @Override
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables, Map<String, String> passwordVariables,
                                                     @SuppressWarnings("unused") Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = newHashSet();

        if (!isScriptInterpolationOff()) {
	        String replacedScript = replaceAllWithInterpolation(script, variables, unresolvedVariables, freezeEvenIfUnresolved);

	        if (script != null && !script.equals(replacedScript)) {
		        //something got replaced so can now see what variables were used.
		        final Set<String> variablesUsed = VariableHelper.collectVariables(script);
                final Set<String> variablesInNonInterpolatableValues = getRelease().getVariablesKeysInNonInterpolatableVariableValues()
                        .stream().map(v -> VariableHelper.withVariableSyntax(v)).collect(Collectors.toSet());
                variablesUsed.removeAll(variablesInNonInterpolatableValues);
		        changes.addVariablesUsed(variablesUsed);
	        }
	        script = replacedScript;
        }

        return unresolvedVariables;
    }

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        return executeScript(targetId, operation, new ExecuteTaskAction(this));
    }

    @Override
    @PublicApiMember
    public String getScript() {
        return script;
    }

    @PublicApiMember
    public void setScript(String script) {
        this.script = script;
    }

    @Override
    public String getEngine() {
        return JYTHON_ENGINE;
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        if (!isScriptInterpolationOff()) {
            usagePoints.add(new PropertyUsagePoint(this, "script"));
        }
        return usagePoints;
    }
}
