package com.xebialabs.xlrelease.domain;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import com.google.common.collect.ComparisonChain;

import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.xlrelease.domain.status.FlagStatus.OK;

public abstract class PlanItem extends BaseConfigurationItem implements VisitableItem {
    public static final Comparator<PlanItem> BY_TITLE = (left, right) -> {
        String leftTitle = nullToEmpty(left.getTitle());
        String rightTitle = nullToEmpty(right.getTitle());
        return ComparisonChain.start().compare(leftTitle, rightTitle).result();
    };

    @Property(description="The title of the item.")
    protected String title;

    @Property(required = false, description="The description of the item.")
    protected String description;

    @Property(required = false, description="The owner of the item.")
    protected String owner;

    @Property(required = false, description="The date that the item is supposed to start.")
    protected Date scheduledStartDate;

    @Property(required = false, description="The date that the item is supposed to end.")
    protected Date dueDate;

    @Property(required = false,  description="The actual start date.")
    protected Date startDate;

    @Property(required = false, description="The actual end date.")
    protected Date endDate;

    @Property(required = false, description="The time that the item is supposed to take to complete, in seconds.")
    protected Integer plannedDuration;

    @Property(description="Flags indicate that an item needs attention.", defaultValue="OK")
    protected FlagStatus flagStatus = FlagStatus.OK;

    @Property(required = false, description="The reason the item is flagged.")
    protected String flagComment;

    @Property(required = false, category = "internal")
    protected boolean overdueNotified;

    private final static String negativePlanDurationMessage = "Planned duration cannot be negative";

    @PublicApiMember
    public String getTitle() {
        return title;
    }

    @PublicApiMember
    public void setTitle(String title) {
        this.title = title;
    }

    @PublicApiMember
    public String getDescription() {
        return description;
    }

    @PublicApiMember
    public void setDescription(String value) {
        this.description = value;
    }

    @PublicApiMember
    public String getOwner() {
        return owner;
    }

    @PublicApiMember
    public void setOwner(String value) {
        this.owner = value;
    }

    public boolean hasOwner() {
        return owner != null;
    }

    public boolean hasOwner(String releaseOwner) {
        return getOwner() != null && getOwner().equalsIgnoreCase(releaseOwner);
    }

    @PublicApiMember
    public Date getDueDate() {
        return dueDate;
    }

    @PublicApiMember
    public void setDueDate(Date value) {
        this.dueDate = value;
    }

    @PublicApiMember
    public Date getStartDate() {
        return startDate;
    }

    @PublicApiMember
    public void setStartDate(Date value) {
        this.startDate = value;
    }

    @PublicApiMember
    public Date getScheduledStartDate() {
        return scheduledStartDate;
    }

    @PublicApiMember
    public void setScheduledStartDate(Date scheduledStartDate) {
        this.scheduledStartDate = scheduledStartDate;
    }

    @PublicApiMember
    public Date getEndDate() {
        return endDate;
    }

    @PublicApiMember
    public void setEndDate(Date value) {
        this.endDate = value;
    }

    @PublicApiMember
    public Integer getPlannedDuration() {
        return plannedDuration;
    }

    @PublicApiMember
    public void setPlannedDuration(Integer plannedDuration) {
        this.plannedDuration = plannedDuration;
    }

    public boolean hasPlannedDuration() {
        return getPlannedDuration() != null;
    }

    @PublicApiMember
    public FlagStatus getFlagStatus() {
        return flagStatus;
    }

    @PublicApiMember
    public void setFlagStatus(FlagStatus flagStatus) {
        this.flagStatus = flagStatus;
    }

    @PublicApiMember
    public String getFlagComment() {
        return flagComment;
    }

    @PublicApiMember
    public void setFlagComment(String flagComment) {
        this.flagComment = flagComment;
    }

    public boolean isFlagged() {
        return OK != flagStatus;
    }

    public void checkDatesValidity(Date scheduledStartDate, Date dueDate, Integer plannedDuration) {
        checkArgument(plannedDuration == null || plannedDuration >= 0L, negativePlanDurationMessage);

        if (scheduledStartDate != null && dueDate != null && plannedDuration != null) {
            checkArgument((dueDate.getTime() - scheduledStartDate.getTime()) / 1000L == plannedDuration, "start date and due date must be consistent with the duration");
        }
    }

    public void updateDates(Date scheduledStartDate, Date dueDate, Integer plannedDuration) {
        checkArgument(plannedDuration == null || plannedDuration >= 0L, negativePlanDurationMessage);

        Date scheduledOrStartDate = hasBeenStarted() ? getStartDate(): scheduledStartDate;
        if (scheduledOrStartDate != null && dueDate != null && plannedDuration != null) {
            checkArgument((dueDate.getTime() - scheduledOrStartDate.getTime()) / 1000L == plannedDuration, "start date and due date must be consistent with the duration");
        }

        if (!hasBeenStarted()) {
            setScheduledStartDate(scheduledStartDate);
        }
        setDueDate(dueDate);
        setPlannedDuration(plannedDuration);
    }

    /**
     * Updates a Duration and Due Date.
     *
     * @param plannedDuration time supposed to complete item, in seconds.
     */
    @PublicApiMember
    public void updateDuration(Integer plannedDuration) {
        checkArgument(plannedDuration >= 0L, negativePlanDurationMessage);

        setPlannedDuration(plannedDuration);
        DateTime newDueDate = new DateTime(getStartOrScheduledDate()).plusSeconds(plannedDuration);
        setDueDate(newDueDate.toDate());
    }

    public Changes moveChildren(int offsetInSeconds) {
        Changes changes = new Changes();

        for (PlanItem item : getChildren()) {
            changes.addAll(item.setDates(offsetInSeconds));
        }

        return changes;
    }

    public boolean hasStartOrScheduledDate() {
        return getStartOrScheduledDate() != null;
    }

    public Date getStartOrScheduledDate() {
        if (hasStartDate()) {
            return getStartDate();
        } else if (hasScheduledStartDate()) {
            return getScheduledStartDate();
        } else {
            return null;
        }
    }

    public boolean hasEndOrDueDate() {
        return getEndOrDueDate() != null;
    }

    public Date getEndOrDueDate() {
        if (hasEndDate()) {
            return getEndDate();
        } else if (hasDueDate()) {
            return getDueDate();
        } else {
            return null;
        }
    }

    public boolean hasScheduledStartDate() {
        return getScheduledStartDate() != null;
    }

    public boolean hasStartDate() {
        return getStartDate() != null;
    }

    public boolean hasDueDate() {
        return getDueDate() != null;
    }

    public boolean hasEndDate() {
        return getEndDate() != null;
    }

    public Changes setDates(int offsetInSeconds) {
        Changes changes = new Changes();

        if (getScheduledStartDate() != null) {
            DateTime newScheduledStartDate = new DateTime(getScheduledStartDate()).plusSeconds(offsetInSeconds);
            setScheduledStartDate(newScheduledStartDate.toDate());
            changes.update(this);
        }

        if (getDueDate() != null) {
            DateTime newDueStartDate = new DateTime(getDueDate()).plusSeconds(offsetInSeconds);
            setDueDate(newDueStartDate.toDate());
            changes.update(this);
        }

        return changes;
    }

    public boolean hasTitle() {
        return isNotEmpty(getTitle());
    }

    public boolean hasTitle(String candidateTitle) {
        return title.equals(candidateTitle);
    }

    public boolean isTitleContaining(String candidateTitle) {
        if (title == null) return true;
        if (candidateTitle == null) return true;

        return title.contains(candidateTitle);
    }

    public boolean hasValidStartDates() {
        if (hasScheduledStartDate() && hasDueDate()) {
            return new DateTime(getScheduledStartDate()).isBefore(new DateTime(getDueDate()));
        }

        return true;
    }

    public abstract List<PlanItem> getChildren();

    public void setStartAndEndDatesIfEmpty() {
        if (!hasStartDate()) {
            setStartDate(new Date());
        }
        if (!hasEndDate()) {
            setEndDate(new Date());
        }
    }

    public boolean isOverdue() {
        Optional<Date> optional = getOrCalculateDueDate();
        return optional.map(date -> date.before(new Date())).orElse(false);
    }

    public void setOverdueNotified(boolean overdueNotified) {
        this.overdueNotified = overdueNotified;
    }

    public boolean isOverdueNotified() {
        return overdueNotified;
    }

    public boolean shouldNotifyOverdue() {
        return !overdueNotified && isOverdue();
    }

    public abstract boolean hasBeenStarted();

    public abstract boolean isDone();

    public abstract Release getRelease();

    public abstract Integer getReleaseUid();

    public abstract void setReleaseUid(Integer releaseUid);

    public abstract boolean isUpdatable();

    public abstract String getDisplayPath();

    public abstract boolean isAborted();

    public abstract boolean isActive();

    public abstract List<UsagePoint> getVariableUsages();

    protected Optional<Date> calculateDueDate(Integer plannedDuration) {
        if (!hasStartDate() || plannedDuration == null) {
            return Optional.empty();
        }

        DateTime dueDateTime = new DateTime(startDate).plusSeconds(plannedDuration);
        return Optional.ofNullable(dueDateTime.toDate());
    }

    public Optional<Date> getOrCalculateDueDate() {
        if (hasDueDate()) {
            return Optional.ofNullable(dueDate);
        } else if (hasPlannedDuration()) {
            return calculateDueDate(plannedDuration);
        }
        return Optional.empty();
    }

    public Duration getComputedPlannedDuration() {
        if (hasPlannedDuration()) {
            return Duration.standardSeconds(getPlannedDuration());
        } else if (hasDueDate() && hasScheduledStartDate()) {
            LocalDateTime localScheduledStartDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(getScheduledStartDate().getTime()), ZoneId.systemDefault());
            LocalDateTime localDueDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(getDueDate().getTime()), ZoneId.systemDefault());
            return Duration.millis(localScheduledStartDate.until(localDueDate, ChronoUnit.MILLIS));
        } else {
            return null;
        }
    }

    public Duration getActualDuration() {
        if (hasStartDate()) {
            LocalDateTime startDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(getStartDate().getTime()), ZoneId.systemDefault());
            if (hasEndDate()) {
                LocalDateTime endDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(getEndDate().getTime()), ZoneId.systemDefault());
                return Duration.millis(startDate.until(endDate, ChronoUnit.MILLIS));
            } else {
                LocalDateTime endDate = LocalDateTime.now();
                return Duration.millis(startDate.until(endDate, ChronoUnit.MILLIS));
            }
        } else {
            return null;
        }
    }

}
