package com.xebialabs.xlrelease.domain;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.utils.PythonScriptCiHelper;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.xlrelease.domain.ScriptTask.JYTHON_ENGINE;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAll;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;
import static java.util.stream.Collectors.toList;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Python Script", versioned = false)
public class CustomScriptTask extends BaseScriptTask {

    public static final Type UNKNOWN_TYPE = Type.valueOf("xlrelease.UnknownType");

    public static final String PYTHON_SCRIPT_PREFIX = "pythonScript.";

    public static final String IGNORE_SCRIPT_VARIABLE_INTERPOLATION = "scriptIgnoreVariableInterpolation";

    public static final String SCRIPT_PROPERTY_NAME = "script";

    public static final String WAIT_FOR_SIGNAL_PROPERTY_NAME = "waitForSignal";

    @Property
    private PythonScript pythonScript;

    @Property(required = false, description = "Status line of the task that will appear in UI.")
    private String statusLine;

    @Property(category = "internal", description = "Path of the next script to schedule.")
    private String nextScriptPath;

    @Property(required = false, category = "internal", description = "Delay in seconds before the next script will start executing.")
    private Integer interval;

    @Property(required = false, description = "Keep the previous output properties on retry")
    private boolean keepPreviousOutputPropertiesOnRetry;

    @Override
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables, Map<String, String> passwordVariables,
                                                     Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = newHashSet();
        PythonScript pythonScript = getPythonScript();
        Collection<PropertyDescriptor> propertiesWithVariables = pythonScript.getPropertiesWithVariables();
        final Set<String> variablesInNonInterpolatableValues = getRelease().getVariablesKeysInNonInterpolatableVariableValues()
                .stream().map(v -> VariableHelper.withVariableSyntax(v)).collect(Collectors.toSet());
        for (PropertyDescriptor propertyDescriptor : propertiesWithVariables) {
            final Object value = pythonScript.getProperty(propertyDescriptor.getName());
            Object newValue = value;
            if (propertyDescriptor.isPassword()) {
                newValue = replaceAll(value, passwordVariables, unresolvedVariables, freezeEvenIfUnresolved);
            } else if (!propertyDescriptor.getName().equals(SCRIPT_PROPERTY_NAME)
                    || (propertyDescriptor.getName().equals(SCRIPT_PROPERTY_NAME) && !isPropertyVariableInterpolationOff(getPythonScript()))) {
                newValue = replaceAllWithInterpolation(value, variables, unresolvedVariables, freezeEvenIfUnresolved);
            }
            pythonScript.setProperty(propertyDescriptor.getName(), newValue);

            if (value != null && !value.equals(newValue)) {
	            //something got replaced so can now see what variables were used.
	            final Set<String> variablesUsed = VariableHelper.collectVariables(value);
                variablesUsed.removeAll(variablesInNonInterpolatableValues);
	            changes.addVariablesUsed(variablesUsed);
            }
        }
        if (!propertiesWithVariables.isEmpty()) {
            changes.update(pythonScript);
        }

        return unresolvedVariables;
    }

    @Override
    protected boolean shouldFreezeVariableMapping(final CiProperty property) {
        return !CATEGORY_OUTPUT.equals(property.getDescriptor().getCategory());
    }

    @Override
    public boolean isPreconditionEnabled() {
        return pythonScript.getProperty("preconditionEnabled");
    }

    @Override
    public boolean isFailureHandlerEnabled() {
        return pythonScript.getProperty("failureHandlerEnabled");
    }

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        return executeScript(targetId, operation, new ExecuteTaskAction(this));
    }

    @PublicApiMember
    public PythonScript getPythonScript() {
        return pythonScript;
    }

    @PublicApiMember
    public void setPythonScript(PythonScript pythonScript) {
        this.pythonScript = pythonScript;
    }

    @Override
    public String getEngine() {
        return JYTHON_ENGINE;
    }

    @PublicApiMember
    public String getStatusLine() {
        return statusLine;
    }

    public void setStatusLine(final String statusLine) {
        this.statusLine = statusLine;
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        usagePoints.addAll(getPythonScript().getInputProperties().stream()
                .filter(pd -> !pd.getName().equals(SCRIPT_PROPERTY_NAME)
                        || (pd.getName().equals(SCRIPT_PROPERTY_NAME) && !isPropertyVariableInterpolationOff(pythonScript)))
                .map(pd -> new PropertyUsagePoint(getPythonScript(), pd.getName())).collect(toList()));
        usagePoints.addAll(getPythonScript().getOutputProperties().stream()
                .map(pd -> new PropertyUsagePoint(getPythonScript(), pd.getName())).collect(toList()));
        return usagePoints;
    }

    public void setNextScriptPath(final String nextScriptPath) {
        this.nextScriptPath = nextScriptPath;
    }

    public String getNextScriptPath() {
        return nextScriptPath;
    }

    public Integer getInterval() {
        return interval;
    }

    public void setInterval(final Integer interval) {
        this.interval = interval;
    }

    public boolean hasNextScriptToExecute() {
        return Strings.isNotBlank(nextScriptPath);
    }

    /**
     * Tells the custom script task that the next script needs to be scheduled after this Jython script is finished.
     * The passed script path must be a valid Jython script on the classpath.
     */
    @PublicApiMember
    public void schedule(String scriptPath) {
        schedule(scriptPath, null);
    }

    /**
     * Tells the custom script task that the next script needs to be scheduled after this Jython script is finished with specified delay in seconds.
     * The passed script path must be a valid Jython script on the classpath.
     * The passed interval must be any value between 1 and 2<sup>31</sup>-1.
     */
    @PublicApiMember
    public void schedule(String scriptPath, Integer interval) {
        if (interval != null && interval < 1) {
            throw new IllegalArgumentException("interval must be in range between 1 and " + Integer.MAX_VALUE);
        }
        this.nextScriptPath = scriptPath;
        this.interval = interval;
    }

    public void resetSchedule() {
        this.nextScriptPath = null;
        this.interval = null;
    }

    @Override
    public Changes fail(String targetId, String failReason, User user, boolean fromAbort) {
        Changes changes = super.fail(targetId, failReason, user, fromAbort);
        resetSchedule();
        changes.update(this);
        return changes;
    }

    public boolean isWaitingForSignal() {
        return pythonScript.hasProperty(WAIT_FOR_SIGNAL_PROPERTY_NAME) &&
                pythonScript.<Boolean>getProperty(WAIT_FOR_SIGNAL_PROPERTY_NAME);
    }

    @Override
    public Changes retry(final String targetId) {
        resetPythonScriptState();
        statusLine = null;
        Changes changes = super.retry(targetId);
        changes.update(pythonScript);
        return changes;
    }

    private void resetPythonScriptState() {
        resetSchedule();
        getPythonScript().getTransitionalAndOutputProperties()
                .stream()
                .filter(pd -> !isKeepPreviousOutputPropertiesOnRetry() || PythonScriptCiHelper.isEmpty(pythonScript.getProperty(pd.getName())))
                .forEach(
                        propertyDescriptor -> propertyDescriptor.set(getPythonScript(), propertyDescriptor.getDefaultValue())
                );
    }

    public String getScriptPath() {
        if (hasNextScriptToExecute()) {
            return nextScriptPath;
        }
        String customScriptLocation = pythonScript.getProperty(ScriptHelper.SCRIPT_LOCATION_PROPERTY);
        if (Strings.isNotBlank(customScriptLocation)) {
            return customScriptLocation;
        }
        return ScriptHelper.getDefaultPath(pythonScript.getType());
    }

    @Override
    public boolean hasAbortScript() {
        return pythonScript.hasProperty(ScriptHelper.ABORT_SCRIPT_LOCATION_PROPERTY) &&
                Strings.isNotBlank(pythonScript.getProperty(ScriptHelper.ABORT_SCRIPT_LOCATION_PROPERTY));
    }

    public String getAbortScriptPath() {
        if (hasNextScriptToExecute()) {
            return nextScriptPath;
        }
        String abortScriptLocation = pythonScript.getProperty(ScriptHelper.ABORT_SCRIPT_LOCATION_PROPERTY);
        if (Strings.isNotBlank(abortScriptLocation)) {
            return abortScriptLocation;
        }
        return ScriptHelper.getDefaultAbortPath(pythonScript.getType());
    }

    @Override
    public String getAbortScript() throws IOException {
        return pythonScript.getAbortScript();
    }

    public boolean isPropertyVariableInterpolationOff(final PythonScript pythonScript) {
        return pythonScript.hasProperty(IGNORE_SCRIPT_VARIABLE_INTERPOLATION) ? pythonScript.<Boolean>getProperty(IGNORE_SCRIPT_VARIABLE_INTERPOLATION) : false;
    }

    public boolean isUnknown() {
        return pythonScript.getType().equals(UNKNOWN_TYPE);
    }

    @PublicApiMember
    public boolean isKeepPreviousOutputPropertiesOnRetry() {
        return keepPreviousOutputPropertiesOnRetry;
    }

    @PublicApiMember
    public void setKeepPreviousOutputPropertiesOnRetry(final boolean keepPreviousOutputPropertiesOnRetry) {
        this.keepPreviousOutputPropertiesOnRetry = keepPreviousOutputPropertiesOnRetry;
    }

    @Override
    public boolean isSupportedInWorkflow() {
        return pythonScript.getProperty("supportedInWorkflow");
    }
}
