package com.xebialabs.xlrelease.domain;

import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.repository.CiProperty;
import com.xebialabs.xlrelease.risk.domain.RiskProfile;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.xlrelease.repository.Ids.toDomainId;
import static com.xebialabs.xlrelease.variable.VariableHelper.*;
import static java.lang.String.format;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Create Release", versioned = false)
public class CreateReleaseTask extends Task {

    public static final String CREATED_RELEASE_ID = "createdReleaseId";
    public static final String TEMPLATE_ID = "templateId";
    public static final String FOLDER_ID = "folderId";

    @Property(label = "Release title", description = "Title of the release to create.")
    private String newReleaseTitle;

    @Property(label = "Template", description = "Template that will be used to create a release.")
    private String templateId;

    @Property(required = false, category = "internal")
    private String templateIdVariable;

    @Property(label = "Folder", description = "Folder in which to create a release.")
    private String folderId;

    @Property(label = "Start release", required = false, defaultValue = "true", description = "Immediately start the release after it is created.")
    private boolean startRelease = true;

    @Property(asContainment = true, required = false, label = "Variables",
            description = "Variable values to pass to the created release.")
    protected List<Variable> templateVariables = new ArrayList<>();

    @Property(required = false, category = "output", label = "Created release ID", description = "ID of the release created from this task.")
    private String createdReleaseId;

    @Property(required = false, label = "Release tags", description = "Tags that will be propagated to the created release along with Template tags.")
    private Set<String> releaseTags = new HashSet<>();

    @Property(required = false, description = "Risk profile that will be used by the created release")
    private RiskProfile riskProfile;

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        Changes changes = super.execute(targetId, operation);

        if (getStatus() != TaskStatus.IN_PROGRESS) {
            return changes;
        }

        generateExecutionId();
        changes.update(this);

        changes.addPostAction(new ExecuteTaskAction(this));

        return changes;
    }

    @Override
    public Set<String> freezeVariablesInCustomFields(final Map<String, ValueWithInterpolation> variables, final Map<String, String> passwordVariables, final Changes changes, final boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = new HashSet<>();
        setNewReleaseTitle(replaceAllWithInterpolation(getNewReleaseTitle(), variables, unresolvedVariables, freezeEvenIfUnresolved));

        freezeTemplateIdVariable(variables, unresolvedVariables, freezeEvenIfUnresolved);
        freezeFolderIdVariable(variables, unresolvedVariables, freezeEvenIfUnresolved);

        getTemplateVariables().forEach(variable -> {
            Object value = variable.getValue();
            if (variable.isPassword()) {
                value = replaceAll(value, passwordVariables, unresolvedVariables, freezeEvenIfUnresolved);
            } else {
                value = replaceAllWithInterpolation(value, variables, unresolvedVariables, freezeEvenIfUnresolved);
            }
            variable.setUntypedValue(value);
            changes.update(variable);
        });

        @SuppressWarnings("unchecked")
        Set<String> resolvedTags = (Set<String>) replaceAllWithInterpolation(getReleaseTags(), variables, unresolvedVariables, freezeEvenIfUnresolved);
        setReleaseTags(resolvedTags.stream().filter(Strings::isNotBlank).collect(Collectors.toSet()));

        changes.update(this);

        return unresolvedVariables;
    }

    private void freezeTemplateIdVariable(final Map<String, ValueWithInterpolation> variables, Set<String> unresolvedVariables, final boolean freezeEvenIfUnresolved) {
        if (VariableHelper.containsOnlyVariable(getTemplateId())) {
            setTemplateIdVariable(getTemplateId());
        }
        String frozenTemplateId = replaceAllWithInterpolation(getTemplateId(), variables, unresolvedVariables, freezeEvenIfUnresolved);
        if (frozenTemplateId != null) {
            frozenTemplateId = frozenTemplateId.replace('-', '/');
        }
        setTemplateId(frozenTemplateId);
    }

    private void freezeFolderIdVariable(final Map<String, ValueWithInterpolation> variables, Set<String> unresolvedVariables, final boolean freezeEvenIfUnresolved) {
        String frozenFolderId = replaceAllWithInterpolation(getFolderId(), variables, unresolvedVariables, freezeEvenIfUnresolved);
        if (isNotEmpty(frozenFolderId)) {
            frozenFolderId = toDomainId(frozenFolderId);
        }
        setFolderId(frozenFolderId);
    }

    @Override
    protected boolean shouldFreezeVariableMapping(final CiProperty property) {
        return !CATEGORY_OUTPUT.equals(property.getDescriptor().getCategory());
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        usagePoints.add(new PropertyUsagePoint(this, "newReleaseTitle"));
        usagePoints.add(new PropertyUsagePoint(this, TEMPLATE_ID));
        usagePoints.add(new PropertyUsagePoint(this, FOLDER_ID));
        usagePoints.add(new PropertyUsagePoint(this, CREATED_RELEASE_ID));
        for (int i = 0; i < getTemplateVariables().size(); ++i) {
            usagePoints.add(new PropertyUsagePoint(this, format("templateVariables[%s].value", i)));
        }
        return usagePoints;
    }

    @PublicApiMember
    public String getNewReleaseTitle() {
        return newReleaseTitle;
    }

    @PublicApiMember
    public void setNewReleaseTitle(final String newReleaseTitle) {
        this.newReleaseTitle = newReleaseTitle;
    }

    @PublicApiMember
    public String getTemplateId() {
        return templateId;
    }

    @PublicApiMember
    public void setTemplateId(final String templateId) {
        this.templateId = templateId;
    }

    @PublicApiMember
    public String getFolderId() {
        return folderId;
    }

    @PublicApiMember
    public void setFolderId(final String folderId) {
        this.folderId = folderId;
    }

    @PublicApiMember
    public String getCreatedReleaseId() {
        return createdReleaseId;
    }

    public void setCreatedReleaseId(final String createdReleaseId) {
        this.createdReleaseId = createdReleaseId;
    }

    @PublicApiMember
    public List<Variable> getTemplateVariables() {
        return templateVariables;
    }

    @PublicApiMember
    public void setTemplateVariables(List<Variable> variables) {
        this.templateVariables = variables;
    }

    @PublicApiMember
    public boolean getStartRelease() {
        return startRelease;
    }

    @PublicApiMember
    public void setStartRelease(final boolean startRelease) {
        this.startRelease = startRelease;
    }

    @PublicApiMember
    public Set<String> getReleaseTags() {
        return releaseTags;
    }

    @PublicApiMember
    public void setReleaseTags(Set<String> releaseTags) {
        this.releaseTags = releaseTags;
    }

    @PublicApiMember
    public RiskProfile getRiskProfile() {
        return riskProfile;
    }

    @PublicApiMember
    public void setRiskProfile(final RiskProfile riskProfile) {
        this.riskProfile = riskProfile;
    }

    public String getTemplateIdVariable() {
        return templateIdVariable;
    }

    public void setTemplateIdVariable(final String templateIdVariable) {
        this.templateIdVariable = templateIdVariable;
    }

    public Map<String, Variable> getTemplateVariablesByKeys() {
        return indexByKey(templateVariables);
    }

}
