package com.xebialabs.xlrelease.configuration;

import com.google.common.base.Preconditions;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlrelease.domain.BaseConfiguration;

import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;

@Metadata(root = CONFIGURATION, versioned = false)
public class ArchivingSettings extends BaseConfiguration {

    public static final String ARCHIVING_SETTINGS_ID = "Configuration/settings/ArchivingSettings";

    public static final int DEFAULT_SEARCH_PAGE_SIZE = 20;
    private static final String DEFAULT_SEARCH_PAGE_SIZE_AS_STRING = "20";

    @Property(defaultValue = "720", required = false, label = "Archive (completed and aborted) releases and workflow executions older than")
    private Integer releaseAgeToDeleteFromJcr; // in hours

    @Property(defaultValue = "16 * * * * *", required = false, hidden = true,
            label = "Schedule for the job that checks for releases to archive (advanced)")
    private String archivingJobCronSchedule;

    @Property(defaultValue = "true", required = false, hidden = true, label = "Enable archiving")
    private boolean enabled;

    @Property(defaultValue = "true", required = false, hidden = false, label = "Enable pre-archiving")
    private boolean preArchivingEnabled;

    @Property(defaultValue = DEFAULT_SEARCH_PAGE_SIZE_AS_STRING, required = false, hidden = true,
            label = "Number of archivable releases to retrieve from the repository in a single search query")
    private Integer searchPageSize;

    @Property(defaultValue = "20", required = false, hidden = true,
            label = "Number of seconds that archiving is allowed to run before it is paused until the next time it is scheduled")
    private Integer maxSecondsPerRun;

    @Property(defaultValue = "1", required = false, hidden = true,
            label = "Number of seconds to sleep between archiving individual releases within a single run")
    private Integer sleepSecondsBetweenReleases;

    @Property(defaultValue = "false", required = false, label = "Enable purging of archived releases and workflow executions")
    private boolean purgingEnabled;

    @Property(defaultValue = "purged", required = false, label = "Path relative to installation to store purged releases")
    private String purgeExportPath;

    @Property(defaultValue = "true", required = false, label = "Export release and workflow execution data when purging")
    private boolean exportOnPurgeEnabled;

    // Defaults to 1 year
    @Property(defaultValue = "8760", required = false, label = "Maximum archive retention period")
    private Integer maximumArchiveRetentionPeriod; // In hours

    // Every 10 minutes at the 40 second mark
    @Property(defaultValue = "40 0/10 * * * *", required = false, hidden = true,
            label = "Schedule for the job that checks for releases to purge from the archive (advanced)")
    private String purgingJobCronSchedule;

    @Property(defaultValue = DEFAULT_SEARCH_PAGE_SIZE_AS_STRING, required = false, hidden = true,
            label = "Number of purgable releases to retrieve from the repository in a single search query")
    private Integer purgingSearchPageSize;

    @Property(defaultValue = "30", required = false, hidden = true,
            label = "Number of seconds that purging is allowed to run before it is paused until the next time it is scheduled")
    private Integer purgingMaxSecondsPerRun;

    @Property(defaultValue = "1", required = false, hidden = true,
            label = "Number of seconds to sleep between purging individual releases within a single run")
    private Integer purgingSleepSecondsBetweenReleases;

    @Property(defaultValue = "false", required = false, label = "Allow releases and workflow executions to skip archiving")
    private boolean skipArchiveAllowed;

    @Property(defaultValue = "false", required = false, label = "Enable automatic abort of stale releases and workflow executions")
    private boolean releaseCleanupEnabled;

    @Property(defaultValue = "2160", required = false, label = "Automatically abort long running releases and workflow executions older than")
    private Integer releaseAgeToAbort; // in hours

    @Property(defaultValue = "1", required = false, hidden = true,
            label = "Number of seconds to sleep between aborting individual releases and workflow executions within a single run")
    private Integer releaseCleanupSleepSecondsBetweenAbort;

    @Property(defaultValue = DEFAULT_SEARCH_PAGE_SIZE_AS_STRING, required = false, hidden = true,
            label = "Number of stale releases and workflow executions to retrieve from the repository in a single search query")
    private Integer releaseCleanupSearchPageSize;

    @Property(defaultValue = "20", required = false, hidden = true,
            label = "Number of seconds that releases and workflow executions aborting is allowed to run before it is paused until the next time it is scheduled")
    private Integer releaseCleanupMaxSecondsPerRun;

    @Property(defaultValue = "46 0/5 0-8,20-23 * * *", required = false, hidden = true,
            label = "Schedule for the job that checks for stale releases and workflow executions to abort (advanced)")
    private String releaseCleanupCronSchedule;

    public void validate() {
        Preconditions.checkArgument(releaseAgeToDeleteFromJcr >= 0, "releaseAgeToDeleteFromJcr must be greater than or equal to 0 hours.");
        if (purgingEnabled) {
            Preconditions.checkArgument(maximumArchiveRetentionPeriod >=1, "If purging is enabled the maximumArchiveRetentionPeriod must be greater than or equal to 1 hour" );
        }
        Preconditions.checkArgument(releaseAgeToAbort >= 1, "releaseAgeToAbort must be greater than or equal to 1 hours.");
    }

    public Integer getReleaseAgeToDeleteFromJcr() {
        return releaseAgeToDeleteFromJcr;
    }

    public String getArchivingJobCronSchedule() {
        return archivingJobCronSchedule;
    }

    public void setReleaseAgeToDeleteFromJcr(final Integer releaseAgeToDeleteFromJcr) {
        this.releaseAgeToDeleteFromJcr = releaseAgeToDeleteFromJcr;
    }
    
    public boolean getEnabled() {
        return enabled;
    }

    public boolean getPreArchivingEnabled() {
        return preArchivingEnabled;
    }

    public void setPreArchivingEnabled(Boolean enabled) {
        preArchivingEnabled = enabled;
    }

    public Integer getSearchPageSize() {
        return searchPageSize;
    }

    public Integer getMaxSecondsPerRun() {
        return maxSecondsPerRun;
    }

    public Integer getSleepSecondsBetweenReleases() {
        return sleepSecondsBetweenReleases;
    }

    public boolean getPurgingEnabled() {
        return this.purgingEnabled;
    }

    public void setPurgingEnabled(boolean enabled) {
        this.purgingEnabled = enabled;
    }

    public String getPurgeExportPath() {
        return this.purgeExportPath;
    }

    public void setPurgeExportPath(String location) {
        this.purgeExportPath = location;
    }
    public boolean getExportOnPurgeEnabled() {
        return this.exportOnPurgeEnabled;
    }

    public void setExportOnPurgeEnabled(boolean enabled) {
        this.exportOnPurgeEnabled = enabled;
    }

    public Integer getMaximumArchiveRetentionPeriod() {
        return this.maximumArchiveRetentionPeriod;
    }

    public void setMaximumArchiveRetentionPeriod(Integer maximumArchiveRetentionPeriod) {
        this.maximumArchiveRetentionPeriod = maximumArchiveRetentionPeriod;
    }

    public String getPurgingJobCronSchedule() {
        return this.purgingJobCronSchedule;
    }

    public Integer getPurgingSearchPageSize() {
        return this.purgingSearchPageSize;
    }

    public Integer getPurgingMaxSecondsPerRun() {
        return this.purgingMaxSecondsPerRun;
    }

    public Integer getPurgingSleepSecondsBetweenReleases() {
        return purgingSleepSecondsBetweenReleases;
    }

    public boolean getSkipArchiveAllowed() {
        return this.skipArchiveAllowed;
    }

    public void setSkipArchiveAllowed(boolean skipArchiveAllowed) {
        this.skipArchiveAllowed = skipArchiveAllowed;
    }

    public boolean isReleaseCleanupEnabled() {
        return releaseCleanupEnabled;
    }

    public void setReleaseCleanupEnabled(final boolean releaseCleanupEnabled) {
        this.releaseCleanupEnabled = releaseCleanupEnabled;
    }

    public Integer getReleaseAgeToAbort() {
        return releaseAgeToAbort;
    }

    public void setReleaseAgeToAbort(final Integer releaseAgeToAbort) {
        this.releaseAgeToAbort = releaseAgeToAbort;
    }

    public Integer getReleaseCleanupSleepSecondsBetweenAbort() {
        return releaseCleanupSleepSecondsBetweenAbort;
    }

    public void setReleaseCleanupSleepSecondsBetweenAbort(final Integer releaseCleanupSleepSecondsBetweenAbort) {
        this.releaseCleanupSleepSecondsBetweenAbort = releaseCleanupSleepSecondsBetweenAbort;
    }

    public Integer getReleaseCleanupSearchPageSize() {
        return releaseCleanupSearchPageSize;
    }

    public void setReleaseCleanupSearchPageSize(final Integer releaseCleanupSearchPageSize) {
        this.releaseCleanupSearchPageSize = releaseCleanupSearchPageSize;
    }

    public Integer getReleaseCleanupMaxSecondsPerRun() {
        return releaseCleanupMaxSecondsPerRun;
    }

    public void setReleaseCleanupMaxSecondsPerRun(final Integer releaseCleanupMaxSecondsPerRun) {
        this.releaseCleanupMaxSecondsPerRun = releaseCleanupMaxSecondsPerRun;
    }

    public String getReleaseCleanupCronSchedule() {
        return releaseCleanupCronSchedule;
    }

    public void setReleaseCleanupCronSchedule(final String releaseCleanupCronSchedule) {
        this.releaseCleanupCronSchedule = releaseCleanupCronSchedule;
    }

}
