package com.xebialabs.xlrelease.builder;

import java.util.*;

import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.recover.TaskRecoverOp;
import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.TaskStatus;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.xlrelease.domain.PythonScript.PYTHON_SCRIPT_ID;
import static com.xebialabs.xlrelease.domain.PythonScript.PYTHON_SCRIPT_PROPERTY;
import static java.util.Collections.emptyList;

/**
 * @param <T> the type of task being built
 * @param <S> the "self-type" of the builder (used to conserve specificity when chaining calls)
 */
public abstract class TaskBuilder<T extends Task, S extends TaskBuilder<T, S>> {

    public static DefaultTaskBuilder newTask() {
        return new DefaultTaskBuilder();
    }

    public static GateTaskBuilder newGateTask() {
        return new GateTaskBuilder();
    }

    public static NotificationTaskBuilder newNotificationTask() {
        return new NotificationTaskBuilder();
    }

    public static ScriptTaskBuilder newScriptTask() {
        return new ScriptTaskBuilder();
    }

    public static ParallelGroupBuilder newParallelGroup() {
        return new ParallelGroupBuilder();
    }

    public static SequentialGroupBuilder newSequentialGroup() {
        return new SequentialGroupBuilder();
    }

    public static CustomScriptTaskBuilder newCustomScript(String scriptDefinitionType) {
        return new CustomScriptTaskBuilder(scriptDefinitionType);
    }

    public static ContainerTaskBuilder newContainerTask(String type) {
        return new ContainerTaskBuilder(type);
    }

    public static RemoteScriptExecutionBuilder newRemoteExecution(String type) {
        return new RemoteScriptExecutionBuilder(type);
    }

    public static PythonScript pythonScript(CustomScriptTask task) {
        PythonScript pythonScript = task.getProperty(PYTHON_SCRIPT_PROPERTY);
        pythonScript.setId(task.getId() + "/" + PYTHON_SCRIPT_ID);
        pythonScript.setCustomScriptTask(task);
        return pythonScript;
    }


    public static CustomScriptTaskBuilder newCustomScript(String scriptDefinitionType, String type) {
        return new CustomScriptTaskBuilder(scriptDefinitionType, type);
    }

    public static UserInputTaskBuilder newUserInputTask() {
        return new UserInputTaskBuilder();
    }

    public static CreateReleaseTaskBuilder newCreateReleaseTask() {
        return new CreateReleaseTaskBuilder();
    }

    @SuppressWarnings("unchecked")
    protected final S self() {
        return (S) this;
    }

    private String id;
    private String title;
    private String precondition;
    private String failureHandler;
    private boolean taskFailureHandlerEnabled;
    private boolean checkAttributes;
    private TaskRecoverOp taskRecoverOp;
    private TaskStatus status = TaskStatus.PLANNED;
    private FlagStatus flagStatus = FlagStatus.OK;
    private String flagComment;
    private String owner;
    private String team;
    private TaskContainer container;
    private String description;
    private Date dueDate;
    private Date scheduledStartDate;
    private Date originalScheduledStartDate;
    private Date startDate;
    private Date endDate;
    private Integer plannedDuration;
    private boolean overdueNotified;
    private boolean dueSoonNotified;
    private List<Comment> comments = emptyList();
    private boolean waitForScheduledStartDate = true;
    private boolean delayDuringBlackout = false;
    private boolean postponedDueToBlackout = false;
    private boolean postponedUntilEnvironmentsAreReserved = false;
    private List<Attachment> attachments = newArrayList();
    private int failuresCount = 0;
    private boolean hasBeenDelayed = false;
    private boolean hasBeenFlagged = false;
    private Map<String, String> variableMapping = newHashMap();
    private List<String> tags = newArrayList();
    private boolean locked = false;
    private List<Facet> facets = newArrayList();
    private Set<String> watchers = new HashSet<>();

    public S withId(String id) {
        this.id = id;
        return self();
    }

    public S withTitle(String title) {
        this.title = title;
        return self();
    }

    public S withIdAndTitle(String id) {
        this.id = id;
        this.title = id;
        return self();
    }

    public S withStatus(TaskStatus status) {
        this.status = status;
        if (status == TaskStatus.IN_PROGRESS) {
            this.startDate = new Date();
        } else if (status != null && status.isDone()) {
            this.endDate = new Date();
        }
        return self();
    }

    public S withFlagStatus(FlagStatus flagStatus) {
        this.flagStatus = flagStatus;
        return self();
    }

    public S withFlagComment(String flagComment) {
        this.flagComment = flagComment;
        return self();
    }

    public S withOwner(String owner) {
        this.owner = owner;
        return self();
    }

    public S withTeam(String team) {
        this.team = team;
        return self();
    }

    public S withContainer(TaskContainer item) {
        this.container = item;
        return self();
    }

    public S withDescription(String description) {
        this.description = description;
        return self();
    }

    public S withDueDate(Date dueDate) {
        this.dueDate = dueDate;
        return self();
    }

    public S withStartDate(Date startDate) {
        this.startDate = startDate;
        return self();
    }

    public S withScheduledStartDate(Date scheduledStartDate) {
        this.scheduledStartDate = scheduledStartDate;
        return self();
    }

    public S withOriginalScheduledStartDate(Date originalScheduledStartDate) {
        this.originalScheduledStartDate = originalScheduledStartDate;
        return self();
    }

    public S withEndDate(Date endDate) {
        this.endDate = endDate;
        return self();
    }

    public S withPlannedDuration(Integer plannedDuration) {
        this.plannedDuration = plannedDuration;
        return self();
    }

    public S withComments(Comment... comments) {
        this.comments = newArrayList(comments);
        return self();
    }

    public S withOverdueNotified(boolean isNotified) {
        this.overdueNotified = isNotified;
        return self();
    }

    public S withDueSoonNotified(boolean isNotified) {
        this.dueSoonNotified = isNotified;
        return self();
    }

    public S withWaitForScheduledStartDate(boolean waitForScheduledStartDate) {
        this.waitForScheduledStartDate = waitForScheduledStartDate;
        return self();
    }

    public S withDelayDuringBlackout(boolean delayDuringBlackout) {
        this.delayDuringBlackout = delayDuringBlackout;
        return self();
    }

    public S beenPostponedDueToBlackout() {
        this.postponedDueToBlackout = true;
        return self();
    }

    public S withAttachments(Attachment... attachments) {
        this.attachments = newArrayList(attachments);
        return self();
    }

    public S withFailuresCount(int failuresCount) {
        this.failuresCount = failuresCount;
        return self();
    }

    public S beenDelayed() {
        this.hasBeenDelayed = true;
        return self();
    }

    public S beenFlagged() {
        this.hasBeenFlagged = true;
        return self();
    }

    public S withPrecondition(String script) {
        this.precondition = script;
        return self();
    }

    public S withCheckAttributes(boolean check) {
        this.checkAttributes = check;
        return self();
    }

    public S beenPostponedUntilEnvironmentsAreReserved() {
        this.postponedUntilEnvironmentsAreReserved = true;
        return self();
    }

    public S withFailureHandler(String script) {
        this.failureHandler = script;
        return self();
    }

    public S withTaskFailureHandlerEnabled(boolean enabled) {
        this.taskFailureHandlerEnabled = enabled;
        return self();
    }

    public S withTaskRecoverOp(TaskRecoverOp recoverOp) {
        this.taskRecoverOp = recoverOp;
        return self();
    }

    public S withVariableMapping(Map<String, String> variableMapping) {
        this.variableMapping = newHashMap(variableMapping);
        return self();
    }

    public S withTags(String... tags) {
        this.tags = newArrayList(tags);
        return self();
    }

    public S withLocked(boolean locked) {
        this.locked = locked;
        return self();
    }

    public S isLocked() {
        this.locked = true;
        return self();
    }

    public S withFacets(Facet... facets) {
        this.facets = newArrayList(facets);
        return self();
    }

    public S withWatchers(Set<String> watchers) {
        this.watchers.addAll(watchers);
        return self();
    }

    public TaskBuilder completed() {
        if (this.startDate == null) {
            this.startDate = new Date();
        }
        if (this.endDate == null) {
            this.endDate = new Date();
        }
        this.status = TaskStatus.COMPLETED;
        return self();
    }

    protected void setFields(T task) {
        task.setId(id);
        task.setStatus(status);
        task.setFlagStatus(flagStatus);
        task.setFlagComment(flagComment);
        task.setTitle(title);
        task.setOwner(owner);
        task.setTeam(team);
        task.setContainer(container);
        task.setDescription(description);
        task.setDueDate(dueDate);
        task.setStartDate(startDate);
        task.setScheduledStartDate(scheduledStartDate);
        task.setOriginalScheduledStartDate(originalScheduledStartDate);
        task.setEndDate(endDate);
        task.setPlannedDuration(plannedDuration);
        task.setOverdueNotified(overdueNotified);
        task.setDueSoonNotified(dueSoonNotified);
        task.setAttachments(attachments);
        task.getComments().addAll(comments);
        task.setWaitForScheduledStartDate(waitForScheduledStartDate);
        task.setDelayDuringBlackout(delayDuringBlackout);
        task.setPostponedDueToBlackout(postponedDueToBlackout);
        task.setPostponedUntilEnvironmentsAreReserved(postponedUntilEnvironmentsAreReserved);
        task.setFailuresCount(failuresCount);
        task.setHasBeenDelayed(hasBeenDelayed);
        task.setHasBeenFlagged(hasBeenFlagged);
        task.setPrecondition(precondition);
        task.setFailureHandler(failureHandler);
        task.setTaskFailureHandlerEnabled(taskFailureHandlerEnabled);
        task.setTaskRecoverOp(taskRecoverOp);
        task.setVariableMapping(variableMapping);
        task.setTags(tags);
        task.setLocked(locked);
        task.setFacets(facets);
        task.setCheckAttributes(checkAttributes);
        task.setWatchers(watchers);
    }

    public abstract T build();
}
