package com.xebialabs.xlrelease.builder;

import java.util.Date;
import java.util.List;
import org.joda.time.DateTime;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.status.PhaseStatus;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.domain.status.PhaseStatus.COMPLETED;

public class PhaseBuilder {
    private String id;
    private String title;
    private PhaseStatus status = PhaseStatus.PLANNED;
    private List<Task> tasks;
    private Release release;
    private Date scheduledStartDate;
    private Date startDate;
    private Date dueDate;
    private Date endDate;
    private Integer plannedDuration;
    private String description;
    private String color;
    private String originId;
    private CiAttributes ciAttributes;

    private PhaseBuilder() {}

    public static PhaseBuilder newPhase() {
        return new PhaseBuilder().withRelease(new Release());
    }


    public PhaseBuilder withOriginId(String originId) {
        this.originId = originId;
        return this;
    }

    public PhaseBuilder withId(String id) {
        this.id = id;
        return this;
    }

    public PhaseBuilder withTitle(String title) {
        this.title = title;
        return this;
    }

    public PhaseBuilder withIdAndTitle(String id) {
        this.id = id;
        this.title = id;
        return this;
    }

    public PhaseBuilder withDescription(String description) {
        this.description = description;
        return this;
    }

    public PhaseBuilder withStatus(PhaseStatus status) {
        this.status = status;
        return this;
    }

    public PhaseBuilder withTasks(Task... tasks) {
        this.tasks = newArrayList(tasks);
        return this;
    }

    public PhaseBuilder withRelease(Release release) {
        this.release = release;
        return this;
    }

    public PhaseBuilder withScheduledStartDate(Date date) {
        this.scheduledStartDate = date;
        return this;
    }

    public PhaseBuilder withStartDate(Date date) {
        this.startDate = date;
        return this;
    }

    public PhaseBuilder withStartDate(DateTime date) {
        this.startDate = date.toDate();
        return this;
    }

    public PhaseBuilder withDueDate(Date date) {
        this.dueDate = date;
        return this;
    }

    public PhaseBuilder withEndDate(DateTime date) {
        this.endDate = date.toDate();
        return this;
    }

    public PhaseBuilder completed() {
        if (this.startDate == null) {
            this.startDate = new Date();
        }
        if (this.endDate == null) {
            this.endDate = new Date();
        }
        this.status = COMPLETED;
        return this;
    }

    public PhaseBuilder withEndDate(Date date) {
        this.endDate = date;
        return this;
    }

    public PhaseBuilder withPlannedDuration(Integer plannedDuration) {
        this.plannedDuration = plannedDuration;
        return this;
    }

    public PhaseBuilder withColor(String color) {
        this.color = color;
        return this;
    }

    public PhaseBuilder withCiAttributes(CiAttributes ciAttributes) {
        this.ciAttributes = ciAttributes;
        return this;
    }

    public Phase build() {
        final Phase phase = new Phase().getType().getDescriptor().newInstance(id);
        phase.setOriginId(originId);
        phase.setStatus(status);
        phase.setTitle(title);
        phase.setDescription(description);
        phase.setRelease(release);
        phase.setScheduledStartDate(scheduledStartDate);
        phase.setStartDate(startDate);
        phase.setDueDate(dueDate);
        phase.setEndDate(endDate);
        phase.setPlannedDuration(plannedDuration);
        phase.setColor(color);

        if (ciAttributes != null) {
            phase.set$ciAttributes(ciAttributes);
        }

        if (null != tasks) {
            for (Task task : tasks) {
                task.setContainer(phase);
            }
            phase.setTasks(tasks);
        }
        return phase;
    }
}
