package com.xebialabs.xlrelease.utils;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;

/**
 * A collection of values that need to be removed from the logs. The black-listed values come from passwords and other sensitive values used in
 * non-password fields.
 */
public class SensitiveValueScrubber {

	private static final SensitiveValueScrubber DISABLED_INSTANCE = new SensitiveValueScrubber();
	private final String[] scrubValues;
	private final String[] replacementValue;
	private final String toStringValue;

	/**
	 * The scrubber won't attempt to clean anything because there is nothing to scrub in this case.
	 * @return a null object instance
	 */
//	@VisibleForTesting
	public static SensitiveValueScrubber disabled() {
		return DISABLED_INSTANCE;
	}

	private SensitiveValueScrubber() {
		scrubValues = new String[0];
		replacementValue = new String[0];
		toStringValue = "";
	}

	public SensitiveValueScrubber(final Set<String> variablesUsed, final Map<String, String> variableMap) {
		Set<String> resolvedValues = new HashSet<>();

		for (String variableName : variablesUsed) {
			final String variableValue = variableMap.get(variableName);
			if (variableValue != null) {
				resolvedValues.add(variableValue);
			}
		}

		this.scrubValues = resolvedValues.toArray(new String[0]);
		this.replacementValue = new String[this.scrubValues.length];

		for (int i = 0; i < this.scrubValues.length; i++) {
			replacementValue[i] = "******";
		}
		this.toStringValue = variablesUsed.toString();
	}

	public String scrubValues(String original) {
		if (scrubValues.length == 0 || original.isEmpty()) {
			return original;
		}

		return StringUtils.replaceEachRepeatedly(original, scrubValues, replacementValue);
	}

	@Override
	public String toString() {
		return "SensitiveValuesScrubber(" + toStringValue + ")";
	}
}
