package com.xebialabs.xlrelease.utils;

import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.LazyConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

public class CiHelper {
    private static final Logger logger = LoggerFactory.getLogger(CiHelper.class);

    public static final Function<ConfigurationItem, String> TO_ID = ConfigurationItem::getId;

    public static List<ConfigurationItem> getNestedCis(Collection<? extends ConfigurationItem> cis) {
        Set<ConfigurationItem> visitedItems = new LinkedHashSet<>();
        for (ConfigurationItem ci : cis) {
            visitedItems.addAll(getNestedCis(ci));
        }
        return new ArrayList<>(visitedItems);
    }

    public static List<ConfigurationItem> getNestedCis(ConfigurationItem ci) {
        MessageLogger traceLog = new MessageLogger();
        Set<ConfigurationItem> visitedItems = new LinkedHashSet<>();
        getNestedCis(traceLog, visitedItems, ci);
        return new ArrayList<>(visitedItems);
    }

    private static void getNestedCis(MessageLogger traceLog, Set<ConfigurationItem> alreadyVisitedItems, ConfigurationItem ci) {
        if (!alreadyVisitedItems.contains(ci)) {
            if (logger.isTraceEnabled()) {
                traceLog.log(String.format("Adding ci %s[%s]", ci.getType(), ci));
            }
            alreadyVisitedItems.add(ci);
            List<ConfigurationItem> children = getChildren(traceLog, ci);
            for (ConfigurationItem child : children) {
                traceLog.increaseDepth();
                getNestedCis(traceLog, alreadyVisitedItems, child);
                traceLog.decreaseDepth();
            }
        } else {
            logger.trace("Item {}[{}] was already visited. Circular reference detected. Log: \n {}", ci.getType(), ci, traceLog.msg());
        }
    }


    public static void eraseTokens(ConfigurationItem ci) {
        getNestedCis(ci).forEach(bci -> {
            if (bci instanceof BaseConfigurationItem) {
                ((BaseConfigurationItem) bci).set$token(null);
            }
        });
    }

    public static void rewriteWithNewId(ConfigurationItem ci, String newId) {
        String oldId = ci.getId();
        String oldIdPattern = format("%s(?=/|$)", oldId); // match old CI ID ending with a "/" or end of line

        eraseTokens(ci);
        for (ConfigurationItem nestedCi : getNestedCis(ci)) {
            if (nestedCi.getId() != null && nestedCi.getId().startsWith(oldId)) {
                String rewrittenId = nestedCi.getId().replaceFirst(oldIdPattern, newId);
                nestedCi.setId(rewrittenId);
            }
        }

        for (ConfigurationItem referencedCi : getExternalReferences(ci)) {
            if (referencedCi.getId() != null && referencedCi.getId().startsWith(oldId)) {
                // Redirect CI non-containment references from old children to new children
                String rewrittenId = referencedCi.getId().replaceFirst(oldIdPattern, newId);
                referencedCi.setId(rewrittenId);
            }
        }
    }


    public static Set<ConfigurationItem> getExternalReferences(ConfigurationItem parentCi) {
        Set<ConfigurationItem> references = new HashSet();
        for (ConfigurationItem ci : getNestedCis(parentCi)) {
            for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
                if (property.isAsContainment()) {
                    continue;
                }
                Object value = property.get(ci);
                if (value == null) {
                    continue;
                }
                PropertyKind kind = property.getKind();
                if (kind == SET_OF_CI || kind == LIST_OF_CI) {
                    //noinspection unchecked
                    references.addAll((Collection<ConfigurationItem>) value);
                }
                if (kind == CI && !isChildViaOneOfChildProperties((ConfigurationItem) value, ci)) {
                    references.add((ConfigurationItem) value);
                }
            }
        }
        return references;
    }

    public static boolean isLazyConfigurationItem(ConfigurationItem possibleChild) {
        return possibleChild instanceof LazyConfigurationItem;
    }

    public static void stripChildrenCis(ConfigurationItem ci) {
        for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
            PropertyKind kind = property.getKind();

            if (kind == SET_OF_CI || kind == LIST_OF_CI) {
                @SuppressWarnings("unchecked")
                Collection<ConfigurationItem> references = (Collection<ConfigurationItem>) property.get(ci);
                references.clear();
            }
        }
    }

    public static void removeCisWithId(Collection<? extends ConfigurationItem> fromCis, String idToRemove) {
        fromCis.removeIf(ci -> idToRemove.equals(ci.getId()));
    }

    public static void fixUpInternalReferences(ConfigurationItem parent) {
        MessageLogger traceLog = new MessageLogger();
        fixUpInternalReferences(traceLog, parent);
    }

    private static void fixUpInternalReferences(MessageLogger traceLog, ConfigurationItem parent) {
        List<ConfigurationItem> children = getChildren(traceLog, parent);
        for (ConfigurationItem child : children) {
            for (PropertyDescriptor property : child.getType().getDescriptor().getPropertyDescriptors()) {
                PropertyKind kind = property.getKind();

                if (kind == CI && property.isAsContainment() && property.get(child) == null) {
                    property.set(child, parent);
                }
            }

            fixUpInternalReferences(traceLog, child);
        }
    }

    private static List<ConfigurationItem> getChildren(MessageLogger traceLog, ConfigurationItem ci) {
        List<ConfigurationItem> children = new ArrayList<>();
        if (ci == null) {
            return children;
        }
        for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
            PropertyKind kind = property.getKind();

            if (kind == SET_OF_CI || kind == LIST_OF_CI) {
                Collection<ConfigurationItem> childrenForProperty = getChildrenForProperty(ci, property);
                if (logger.isTraceEnabled()) {
                    childrenForProperty.forEach(child ->
                            traceLog.log(String.format("- found child of %s.%s[%s] => %s[%s] ", ci.getType(), property.getName(), ci, child.getType(), child)));
                }
                children.addAll(childrenForProperty);
            }

            if (kind == CI) {
                ConfigurationItem child = (ConfigurationItem) property.get(ci);
                if (child != null && (isChildViaOneToOneRelationship(child, ci, property) || isNestedProperty(child, ci, property))) {
                    if (logger.isTraceEnabled()) {
                        traceLog.log(String.format("- found child of %s.%s[%s] => %s[%s] ", ci.getType(), property.getName(), ci, child.getType(), child));
                    }
                    children.add(child);
                }
            }
        }
        return children;
    }

    private static boolean isNestedProperty(ConfigurationItem child, ConfigurationItem ci, PropertyDescriptor property) {
        ConfigurationItem value = (ConfigurationItem) property.get(ci);
        return child.equals(value) && property.isNested();
    }

    private static Collection<ConfigurationItem> getChildrenForProperty(ConfigurationItem parent, PropertyDescriptor property) {
        @SuppressWarnings("unchecked")
        Collection<ConfigurationItem> references = (Collection<ConfigurationItem>) property.get(parent);
        if (references == null) {
            return new ArrayList<>();
        }
        Collection<ConfigurationItem> nonNullReferences = references.stream().filter(Objects::nonNull).collect(toList());

        // Parent/child relationship is modeled on the parent
        if (property.isAsContainment()) {
            return nonNullReferences;
        }

        // Parent/child relationship is modeled on the child
        return nonNullReferences.stream()
                .filter(ci -> isChildViaOneOfChildProperties(ci, parent))
                .collect(toList());
    }

    public static boolean isChildViaOneOfChildProperties(ConfigurationItem possibleChild, ConfigurationItem parent) {
        // lazy configuration items are not considered children of a parent
        if (isLazyConfigurationItem(possibleChild)) {
            return false;
        }
        for (PropertyDescriptor property : possibleChild.getType().getDescriptor().getPropertyDescriptors()) {
            PropertyKind kind = property.getKind();

            if (kind == CI && property.isAsContainment() && parent.equals(property.get(possibleChild))) {
                return true;
            }
        }
        return false;
    }

    private static boolean isChildViaOneToOneRelationship(ConfigurationItem possibleChild, ConfigurationItem parent, PropertyDescriptor parentProperty) {
        // in a 1:1 relationship ONLY child has 'as-containment' property set
        boolean parentPropertyIsNotAContainment = !parentProperty.isAsContainment();
        if (parentPropertyIsNotAContainment) {
            for (PropertyDescriptor childProperty : possibleChild.getType().getDescriptor().getPropertyDescriptors()) {
                PropertyKind kind = childProperty.getKind();

                if (kind == CI && childProperty.isAsContainment()) {
                    // One-to-one parent-child relationships do not exist in platform's type system, but we have/need it for
                    // relationships like [CustomScriptTask.pythonScript <-> PythonScript.customScriptTask] or
                    // [Variable.valueProvider <-> ValueProviderConfiguration.variable]

                    if (!(possibleChild instanceof LazyConfigurationItem) || ((LazyConfigurationItem) possibleChild).isInitialized()) {
                        Object possibleParentReference = childProperty.get(possibleChild);
                        if (parent.equals(possibleParentReference)) {
                            // The parent is already set in the asContainment property, so it is the parent
                            return true;
                        }
                        if (possibleParentReference == null
                                && possibleChild.getType().instanceOf(parentProperty.getReferencedType())
                                && parent.getType().instanceOf(childProperty.getReferencedType())) {
                            // There is no parent set but it could be set there. This can happen
                            return true;
                        }
                    }

                }
            }
        }
        return false;
    }

    public static ConfigurationItem forFields(ConfigurationItem ci, PropertyFilter filter, PropertyAction action) {
        ci.getType().getDescriptor().getPropertyDescriptors().stream()
                .filter(filter)
                .forEach(pd -> action.execute(ci, pd));
        return ci;
    }

    @FunctionalInterface
    public interface PropertyFilter extends Predicate<PropertyDescriptor> {
    }

    @FunctionalInterface
    public interface PropertyAction {
        void execute(ConfigurationItem ci, PropertyDescriptor pd);
    }
}

