package com.xebialabs.xlrelease.repository;

import java.util.regex.Pattern;

import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;
import static java.lang.String.format;

public class Ids {

    public static final String ROOT_FOLDER_ID = APPLICATIONS.getRootNodeName();

    public static final String SEPARATOR = "/";

    public static final String RELEASE_PREFIX = "Release";
    public static final String TRIGGER_PREFIX = "Trigger";
    public static final String PHASE_PREFIX = "Phase";
    public static final String TASK_PREFIX = "Task";
    public static final String VARIABLE_PREFIX = "Variable";
    public static final String CONFIGURATION_PREFIX = "Configuration";
    public static final String VALUE_PROVIDER_SUFFIX = "valueProvider";
    public static final String DEPENDENCY_PREFIX = "Dependency";
    public static final String FOLDER_PREFIX = "Folder";
    public static final String ATTACHMENT_PREFIX = "Attachment";
    public static final String TEAM_PREFIX = "Team";
    public static final String DELIVERY_PREFIX = "Delivery";
    private static final String STAGE_PREFIX = "Stage";
    private static final String FACET_PREFIX = "Facet";
    private static final String DASHBOARD_PREFIX = "Dashboard";


    public static final String CUSTOM_CONFIGURATION_ROOT = CONFIGURATION.getRootNodeName() + "/Custom";
    public static final String CALENDAR_ROOT = CONFIGURATION.getRootNodeName() + "/Calendar";
    public static final String MAIL_CONFIGURATION_ROOT = CONFIGURATION.getRootNodeName() + "/mail";
    public static final String RISK_PROFILE_CONFIGURATION_ROOT = CONFIGURATION.getRootNodeName() + "/riskProfiles";
    public static final String DELIVERIES_ROOT = "Deliveries";

    private static final String DELIVERY_REGEX = "((" + DELIVERIES_ROOT + "/" + DELIVERY_PREFIX + ")|(" + DELIVERY_PREFIX + "))[a-zA-Z0-9]{32}";
    private static final Pattern DELIVERY_PATTERN = Pattern.compile(DELIVERY_REGEX);
    private static final Pattern STAGE_PATTERN = Pattern.compile(DELIVERY_REGEX + "/" + STAGE_PREFIX + "[a-zA-Z0-9]{32}");

    private Ids() {
    }

    public static String getParentId(String id) {
        if (isRoot(id))
            throw new IllegalArgumentException("Not a proper ID for a children: " + id);

        return id.substring(0, id.lastIndexOf(SEPARATOR));
    }

    public static boolean isRoot(String id) {
        return !id.contains(SEPARATOR);
    }

    public static String releaseIdFrom(String id) {
        return findInAncestry(id, RELEASE_PREFIX);
    }

    public static String triggerIdFrom(String id) {
        return findInAncestry(id, TRIGGER_PREFIX);
    }

    public static String deliveryIdFrom(String id) {
        return Ids.findInAncestry(id, DELIVERY_PREFIX);
    }

    public static String phaseIdFrom(String id) {
        return findInAncestry(id, PHASE_PREFIX);
    }

    public static String taskIdFrom(String id) {
        return findInAncestry(id, TASK_PREFIX);
    }

    public static boolean isNullId(String id) {
        return id == null || id.trim().isEmpty() || "null".equals(id.trim().toLowerCase());
    }

    public static boolean isPlanItemId(String id) {
        return isReleaseId(id) || isPhaseId(id) || isTaskId(id);
    }

    public static boolean isReleaseId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(RELEASE_PREFIX);
    }

    public static boolean isTriggerId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(TRIGGER_PREFIX);
    }

    public static boolean isPhaseId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(PHASE_PREFIX);
    }

    public static boolean isAttachmentId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(ATTACHMENT_PREFIX);
    }

    public static boolean isTaskId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(TASK_PREFIX);
    }

    public static boolean isVariableId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(VARIABLE_PREFIX);
    }

    public static boolean isDependencyId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(DEPENDENCY_PREFIX);
    }

    public static boolean isFolderId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(FOLDER_PREFIX);
    }

    public static boolean isFacetId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(FACET_PREFIX);
    }

    public static boolean isCustomConfigurationId(String id) {
        checkIdNotNull(id);
        return id.startsWith(CUSTOM_CONFIGURATION_ROOT);
    }

    public static boolean isConfigurationId(String id) {
        checkIdNotNull(id);
        return id.startsWith(CONFIGURATION.getRootNodeName()) || (
                id.startsWith(APPLICATIONS.getRootNodeName()) && getName(id).startsWith(CONFIGURATION_PREFIX)); // check if start with Application but then ends with Config (and in between FOlder)
    }

    public static boolean isTeamId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(TEAM_PREFIX);
    }

    public static boolean isDomainId(String id) {
        checkIdNotNull(id);
        return id.startsWith(APPLICATIONS.getRootNodeName());
    }

    public static boolean isValueProviderId(String id) {
        checkIdNotNull(id);
        int i = id.lastIndexOf('/');

        if (i < 1) {
            return false;
        }

        String[] stringArray = {id.substring(0, i), id.substring(i + 1)};
        return isVariableId(stringArray[0]) && stringArray[1].equals(VALUE_PROVIDER_SUFFIX);
    }

    public static boolean isCalendarId(String id) {
        checkIdNotNull(id);
        return id.startsWith(CALENDAR_ROOT);
    }

    public static boolean isDeliveryId(String id) {
        checkIdNotNull(id);
        return DELIVERY_PATTERN.matcher(id).matches();
    }

    public static boolean isStageId(String id) {
        checkIdNotNull(id);
        return STAGE_PATTERN.matcher(id).matches();
    }

    public static boolean isDashboardId(String id) {
        checkIdNotNull(id);
        return getName(id).startsWith(DASHBOARD_PREFIX);
    }

    public static boolean isInFolder(String id) {
        return !isRoot(id) && isFolderId(getParentId(id));
    }

    public static boolean isInRootFolder(String id) {
        return !isNullId(id) && ROOT_FOLDER_ID.equals(getParentId(id));
    }

    public static boolean hasReleaseId(String id) {
        try {
            return releaseIdFrom(id) != null;
        } catch (IllegalArgumentException ex) {
            return false;
        }
    }

    public static boolean isInRelease(String id) {
        try {
            return !isReleaseId(id) && releaseIdFrom(id) != null;
        } catch (IllegalArgumentException ex) {
            return false;
        }
    }

    public static String findInAncestry(String id, String prefix) {
        if (!id.contains(prefix)) { // avoid potential stack overflow.
            throw new IllegalArgumentException(format("ID '%s' does not contain prefix '%s'", id, prefix));
        }
        try {
            String ancestryId = id;
            while (!getName(ancestryId).startsWith(prefix)) {
                ancestryId = getParentId(ancestryId);
            }
            return ancestryId;
        } catch (Exception ex) {
            throw new IllegalArgumentException(format("Unable to find ancestor ID of the following ID: '%s' and prefix: '%s'", id, prefix), ex);
        }
    }

    private static void checkIdNotNull(final String id) {
        if (id == null)
            throw new IllegalArgumentException("ID may not be null");
    }

    /**
     * Given Application/Release01/Phase01, returns Phase01
     */
    public static String getName(String id) {
        if (id == null || !id.contains(SEPARATOR))
            return id;
        return id.substring(id.lastIndexOf(SEPARATOR) + 1);
    }

    public static String getReleaselessChildId(String childId) {
        if (isReleaseId(childId)) {
            return null;
        }
        return childId.substring(releaseIdFrom(childId).length() + 1);
    }

    public static String getFolderlessId(String id) {
        String folderId = findFolderId(id);
        if (isFolderId(folderId) || ROOT_FOLDER_ID.equals(folderId)) {
            return relativePathFrom(folderId, normalizeId(id));
        }
        return normalizeId(id);
    }

    public static String relativePathFrom(String fromPath, String fullPath) {
        if (fullPath.equals(fromPath)) {
            return "";
        } else if (fullPath.startsWith(fromPath)) {
            return fullPath.substring(fromPath.length() + 1);
        } else {
            throw new IllegalArgumentException(String.format("'%s' does not contain '%s'", fullPath, fromPath));
        }
    }

    public static String getFullTaskName(String id) {
        return id.substring(phaseIdFrom(id).length() + 1);
    }

    public static String getTopLevelTaskId(String id) {
        String topLevelTaskId = id;
        while (isTaskId(getParentId(topLevelTaskId))) {
            topLevelTaskId = getParentId(topLevelTaskId);
        }
        return topLevelTaskId;
    }

    public static String releasePathInRootFolder(String releaseId) {
        checkIdNotNull(releaseId);
        return releaseId.replace(ROOT_FOLDER_ID + SEPARATOR, "");
    }

    public static String findFolderId(String id) {
        String parentId = normalizeId(id);
        while (!Ids.isFolderId(parentId) && !Ids.isRoot(parentId)) {
            parentId = Ids.getParentId(parentId);
        }
        return parentId;
    }

    public static String findCommonParent(String id1, String id2) {
        String[] path1 = id1.split("/");
        String[] path2 = id2.split("/");
        StringBuilder parent = new StringBuilder();
        for (int i = 0; i < path1.length; i++) {
            String item1 = path1[i];
            if (i == path2.length) break;
            String item2 = path2[i];
            if (!item1.equals(item2)) break;
            parent.append(item1).append("/");
        }
        return parent.length() > 0 ? parent.toString().substring(0, parent.length() - 1) : "";
    }

    public static String toDomainId(String id) {
        if (null != id) {
            id = normalizeId(id).replace('-', '/');
            return id.startsWith(ROOT_FOLDER_ID) ? id :
                    ROOT_FOLDER_ID + SEPARATOR + id;
        } else {
            throw new IllegalArgumentException("Id is null while converting to domain id");
        }
    }

    public static String normalizeId(String id) {
        if (id == null) {
            return null;
        }
        return id.startsWith("/") ? id.substring(1) : id;
    }

    public static String formatWithFolderId(String folderId, String containerId) {
        String newFolderId = folderId != null ? folderId : ROOT_FOLDER_ID;
        if (!containerId.startsWith(newFolderId)) {
            return format("%s/%s", newFolderId, containerId);
        } else {
            return containerId;
        }
    }

    public static String releaseActorId(String releaseId) {
        return normalizeId(getFolderlessId(releaseIdFrom(releaseId)));
    }
}
