package com.xebialabs.xlrelease.booter

import com.xebialabs.deployit.booter.local._
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, IDescriptorRegistry}
import com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument
import com.xebialabs.xlplatform.synthetic.yaml.TypeDefinitionYamlDocument
import com.xebialabs.xlplatform.synthetic.{TypeModificationSpecification, TypeSpecification}
import com.xebialabs.xlrelease.booter.XlrBooter._
import com.xebialabs.xlrelease.plugin.Plugin
import com.xebialabs.xlrelease.plugin.classloading.XlrPluginClassLoader

import java.net.URL
import java.util.UUID
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

object XlrBooter {
  final val RELEASE_REGISTRY_NAME = "releaseRegistry"
  final val RELEASE_REGISTRY_ID = HierarchicalDescriptorRegistryId(RELEASE_REGISTRY_NAME)

  private lazy val xlrBooter = {
    val pluginClassLoader = XlrPluginClassLoader()
    val typeSpecificationRepositories = Array[TypeSpecificationRepository](
      new PluginClassLoaderTypeSpecificationRepository(pluginClassLoader)
    )
    new XlrBooter(pluginClassLoader, typeSpecificationRepositories)
  }

  def apply(): XlrBooter = xlrBooter

}

class XlrBooter private(
                         xlrPluginClassLoader: XlrPluginClassLoader,
                         typeSpecificationRepositories: Array[TypeSpecificationRepository]
                       ) {

  def boot(): BootResult = {
    xlrPluginClassLoader.reload()
    val registry = new HierarchicalDescriptorRegistry(RELEASE_REGISTRY_ID, localDescriptorRegistry)
    val typeSpecifications = typeSpecificationRepositories.flatMap(_.find(registry))
    // we have to guard access to DescriptorRegistry while we load descriptorRegistry
    DescriptorRegistry.withWriteLock { () =>
      // descriptorRegistry has to be registered before we register types (inside loadTypes)
      DescriptorRegistry.replaceRegistry(registry)
      loadTypes(registry, typeSpecifications)
      registry.verifyTypes()
      GlobalContextInitializer.init(RELEASE_REGISTRY_ID)
    }
    BootResult(registry, Seq())
  }

  def reboot(): BootResult = {
    // reboot is the same as boot for this booter
    boot()
  }

  def bootWithTempRegistry(plugin: Plugin): BootResult = {
    val registryId = HierarchicalDescriptorRegistryId(UUID.randomUUID().toString)
    val registry = new HierarchicalDescriptorRegistry(registryId, localDescriptorRegistry)
    DescriptorRegistry.add(registry)

    // load type specifications of all plugins EXCEPT the one we're installing right now
    val plugins = xlrPluginClassLoader.getPlugins()
    // merge all type specifications and load them
    val all = plugins.filterNot(_.name() == plugin.name()).collect {
      case p => getTypeSpecifications(p)
    }.toBuffer += getTypeSpecifications(plugin)
    val (validations, typeSpecifications) = all.toSeq.reduceLeft((a, b) => (a._1 ++ b._1, a._2 ++ b._2))
    val verificationResult = try {
      loadTypes(registry, typeSpecifications)
      registry.verifyTypes()
      None
    } catch {
      case ex =>
        Some(ex.getMessage)
    }
    BootResult(registry, validations ++ verificationResult)
  }

  private case class SyntheticData(typeModifications: Seq[TypeModificationSpecification], typeSpecifications: Seq[TypeSpecification])

  private def getXmlTypeData(pluginUrl: URL): SyntheticData = {
    val syntheticXml = SyntheticXmlDocument.read(pluginUrl)
    val typeModifications = syntheticXml.getTypeModifications.asScala
    val xmlTypeSpecifications = syntheticXml.getTypes.asScala
    SyntheticData(typeModifications.toSeq, xmlTypeSpecifications.toSeq)
  }

  private def getYamlTypeData(pluginUrl: URL): SyntheticData = {
    val yamlDocument = TypeDefinitionYamlDocument.read(pluginUrl)
    val yamlTypeSpecifications = yamlDocument.getTypes.asScala
    SyntheticData(Seq.empty[TypeModificationSpecification], yamlTypeSpecifications.toSeq)
  }

  private def getTypeSpecifications(plugin: Plugin): (Seq[String], Seq[TypeSpecification]) = {
    // read type specifications from database
    val validations = ListBuffer[String]()

    val xmlData = plugin.getResources("type-definitions.xml")
      .map(url => getXmlTypeData(url))

    val yamlData = plugin.getResources("type-definitions.yaml")
      .map(url => getYamlTypeData(url))

    val combinedTypeData = (xmlData ++ yamlData)
      .foldLeft(SyntheticData(Seq.empty, Seq.empty)) {
        case (combined, pluginData) =>
          SyntheticData(
            combined.typeModifications ++ pluginData.typeModifications,
            combined.typeSpecifications ++ pluginData.typeSpecifications
          )
      }

    if (combinedTypeData.typeModifications.nonEmpty) {
      validations += s"Plugin ${plugin.name()} contains type modifications - that's not supported in a versioned type system"
    }
    (validations.toSeq, combinedTypeData.typeSpecifications)
  }

  private def loadTypes(descriptorRegistry: HierarchicalDescriptorRegistry, typeSpecifications: Seq[TypeSpecification]): TypeDefinitions = {
    val parentRegistryId = descriptorRegistry.parentDescriptorRegistry.getId
    // initialize our global context with values from parent
    val registryId = descriptorRegistry.getId
    GlobalContextRegistry.copy(parentRegistryId, registryId)
    val parentTypeDefs = TypeDefinitionsRegistry.get(parentRegistryId)
    val existingTypeDefinitions = parentTypeDefs.get.getDefinitions
    val typeDefinitions = new TypeDefinitions(descriptorRegistry, existingTypeDefinitions)
    typeSpecifications.foreach(typeDefinitions.defineType)
    typeDefinitions.registerTypes()
    typeDefinitions
  }

  private def localDescriptorRegistry = {
    DescriptorRegistry.getDescriptorRegistry(LocalDescriptorRegistry.LOCAL_ID).asInstanceOf[LocalDescriptorRegistry]
  }
}

case class BootResult(registry: IDescriptorRegistry, validationErrors: Seq[String]) extends AutoCloseable {
  // once boot result is closed it's not possible to call exists method on a Type as descriptor registry is closed
  override def close(): Unit = registry.close()
}
