package com.xebialabs.xlrelease.plugin

import org.springframework.util.AntPathMatcher

import java.io.File
import java.net.URL
import java.nio.file._
import java.nio.file.attribute.BasicFileAttributes
import scala.collection.mutable.ArrayBuffer
import scala.jdk.CollectionConverters._

case class ExplodedPlugin(explodedDir: File) extends Plugin {
  override def getResources(name: String): Seq[URL] = {
    val f = new File(explodedDir, name)
    if (f.exists()) {
      Seq(f.toURI.toURL)
    } else {
      Seq()
    }
  }

  // no need to close plugins in ext folder cause there's no materialization happening in that folder
  override def close(): Unit = ()

  override def find(pathPattern: String): Seq[URL] = {
    val pathMatcher = new AntPathMatcher()
    val visitOptions = Set[FileVisitOption]().asJava // do not follow links
    val matched: ArrayBuffer[URL] = ArrayBuffer[URL]()
    val visitor: FileVisitor[_ >: Path] = new SimpleFileVisitor[Path] {
      override def visitFile(file: Path, attrs: BasicFileAttributes): FileVisitResult = {
        val res = super.visitFile(file, attrs)
        val filePath = explodedDir.toURI.relativize(file.toUri).toString
        val fileMatched = pathMatcher.`match`(pathPattern, filePath)
        if (fileMatched) {
          matched += file.toAbsolutePath.toUri.toURL
        }
        res
      }
    }
    Files.walkFileTree(explodedDir.toPath, visitOptions, Int.MaxValue, visitor)
    matched.toSeq
  }

  override lazy val name: String = {
    pluginVersionProperties match {
      case Some(props) => props.getProperty("plugin")
      case None => explodedDir.getName
    }
  }

  override def version: String = {
    pluginVersionProperties match {
      case Some(props) => props.getProperty("version")
      case None => "unknown"
    }
  }
}
