package com.xebialabs.xlrelease.plugin.manager.validator

import com.xebialabs.deployit.plugin.api.reflect._
import com.xebialabs.xlrelease.plugin.manager.validator.InputHintComparator.notEqual
import com.xebialabs.xlrelease.utils.Diff

import scala.collection.mutable.ArrayBuffer

sealed trait TypeChangeDetector {
  def diff(oldDescriptors: Seq[Descriptor], newDescriptors: Seq[Descriptor], oldRegistryId: DescriptorRegistryId, newRegistryId: DescriptorRegistryId): Seq[Change]
}

object TypeChangeDetector extends TypeChangeDetector {

  // there is no point in taking default values for CIs as it should not be possible to set those
  // see com.xebialabs.deployit.booter.local.Converters.createStatelessConverter
  private val kindsWithDefaultValueConverter = Set(
    PropertyKind.ENUM,
    PropertyKind.BOOLEAN,
    PropertyKind.INTEGER,
    PropertyKind.STRING,
    PropertyKind.DATE,
    PropertyKind.SET_OF_STRING,
    PropertyKind.LIST_OF_STRING,
    PropertyKind.MAP_STRING_STRING
  )


  import scala.jdk.CollectionConverters._

  def diff(oldDescriptors: Seq[Descriptor], newDescriptors: Seq[Descriptor], oldRegistryId: DescriptorRegistryId, newRegistryId: DescriptorRegistryId): Seq[Change] = {
    val d: Diff[Type, Descriptor] = Diff.applyWithKeyMappingAndComparator(oldDescriptors, newDescriptors)(_.getType, (o, n) => false)
    val deletedTypes: Seq[Change] = d.deletedEntries.collect { case (t, _) => DeletedType(Some(t)) }.toSeq
    val newTypes: Seq[Change] = d.newEntries.collect { case (t, _) => NewType(Some(t)) }.toSeq
    val updates: Seq[Change] = d.updatedEntries.flatMap { case (t, (oldDescriptor, newDescriptor)) => diff(oldDescriptor, newDescriptor, oldRegistryId, newRegistryId) }.toSeq
    Seq.empty[Change] ++ deletedTypes ++ newTypes ++ updates
  }

  def diff(oldDescriptor: Descriptor, newDescriptor: Descriptor, oldRegistryId: DescriptorRegistryId, newRegistryId: DescriptorRegistryId): Option[UpdatedType] = {
    val attributeChanges = diffTypeAttributes(oldDescriptor, newDescriptor)

    val oldPropertyDescriptors = oldDescriptor.getPropertyDescriptors.asScala
    val newPropertyDescriptors = newDescriptor.getPropertyDescriptors.asScala
    val d: Diff[String, PropertyDescriptor] = Diff.applyWithKeyMappingAndComparator(oldPropertyDescriptors, newPropertyDescriptors)(_.getName, (o, n) => false)
    val newProperties: Seq[PropertyChange] = d.newEntries.collect { case (pName, _) => NewProperty(pName) }.toSeq
    val deletedProperties: Seq[PropertyChange] = d.deletedEntries.collect { case (pName, _) => DeletedProperty(pName) }.toSeq
    val updates: Seq[PropertyChange] = d.updatedEntries.flatMap { case (pName, (oldProperty, newProperty)) => diff(pName, oldProperty, newProperty, oldRegistryId, newRegistryId) }.toSeq
    val propertyChanges = Seq.empty[PropertyChange] ++ newProperties ++ deletedProperties ++ updates
    if (propertyChanges.nonEmpty || attributeChanges.nonEmpty) {
      Some(UpdatedType(Some(oldDescriptor.getType), Some(newDescriptor.getType), propertyChanges, attributeChanges))
    } else {
      None
    }
  }

  private def diffTypeAttributes(oldDescriptor: Descriptor, newDescriptor: Descriptor): Seq[AttributeChange[_]] = {
    val attributeChanges = ArrayBuffer[AttributeChange[_]]()

    if (oldDescriptor.isVirtual() != newDescriptor.isVirtual()) {
      attributeChanges += VirtualChange(oldDescriptor.isVirtual, newDescriptor.isVirtual)
    }
    if (oldDescriptor.getLabel() != newDescriptor.getLabel()) {
      attributeChanges += LabelChange(oldDescriptor.getLabel(), newDescriptor.getLabel())
    }
    if (oldDescriptor.getDescription() != newDescriptor.getDescription()) {
      attributeChanges += DescriptionChange(oldDescriptor.getDescription(), newDescriptor.getDescription())
    }

    val oldInterfaces = oldDescriptor.getInterfaces().asScala.toSet[Type].map(t => t.toString())
    val newInterfaces = newDescriptor.getInterfaces().asScala.toSet[Type].map(t => t.toString())

    if (oldInterfaces != newInterfaces) {
      attributeChanges += InterfacesChange(oldInterfaces, newInterfaces)
    }

    val oldSuperClasses = oldDescriptor.getSuperClasses().asScala.toSet[Type].map(t => t.toString())
    val newSuperClasses = newDescriptor.getSuperClasses().asScala.toSet[Type].map(t => t.toString())

    if (oldSuperClasses != newSuperClasses) {
      attributeChanges += SuperClassesChange(oldSuperClasses, newSuperClasses)
    }
    if (oldDescriptor.isVersioned != newDescriptor.isVersioned) {
      attributeChanges += VersionedChange(oldDescriptor.isVersioned, newDescriptor.isVersioned)
    }
    if (oldDescriptor.getContainerType != newDescriptor.getContainerType) {
      attributeChanges += ContainerTypeChange(oldDescriptor.getContainerType, newDescriptor.getContainerType)
    }
    if (oldDescriptor.getDeployableType != newDescriptor.getDeployableType) {
      attributeChanges += DeployableTypeChange(oldDescriptor.getDeployableType, newDescriptor.getDeployableType)
    }
    if (oldDescriptor.isInspectable != newDescriptor.isInspectable) {
      attributeChanges += InspectableChange(oldDescriptor.isInspectable, newDescriptor.isInspectable)
    }
    if (oldDescriptor.getRoot != newDescriptor.getRoot) {
      attributeChanges += RootChange(oldDescriptor.getRoot, newDescriptor.getRoot)
    }
    if (oldDescriptor.getRootName != newDescriptor.getRootName) {
      attributeChanges += RootNameChange(oldDescriptor.getRootName, newDescriptor.getRootName)
    }

    attributeChanges.toSeq
  }

  private def diff(propertyName: String, oldProperty: PropertyDescriptor, newProperty: PropertyDescriptor, oldRegistryId: DescriptorRegistryId, newRegistryId: DescriptorRegistryId): Option[PropertyChange] = {
    val attributeChanges = ArrayBuffer[AttributeChange[_]]()

    if (oldProperty.getDescription != newProperty.getDescription) {
      attributeChanges += DescriptionChange(oldProperty.getDescription, newProperty.getDescription)
    }
    if (oldProperty.isAsContainment != newProperty.isAsContainment) {
      attributeChanges += AsContainmentChange(oldProperty.isAsContainment, newProperty.isAsContainment)
    }
    if (oldProperty.isNested != newProperty.isNested) {
      attributeChanges += NestedChange(oldProperty.isNested, newProperty.isNested)
    }
    if (oldProperty.getCategory != newProperty.getCategory) {
      attributeChanges += CategoryChange(oldProperty.getCategory, newProperty.getCategory)
    }
    if (oldProperty.getLabel != newProperty.getLabel) {
      attributeChanges += LabelChange(oldProperty.getLabel, newProperty.getLabel)
    }
    if (oldProperty.isPassword != newProperty.isPassword) {
      attributeChanges += PasswordChange(oldProperty.isPassword, newProperty.isPassword)
    }
    if (oldProperty.isRequired != newProperty.isRequired) {
      attributeChanges += RequiredChange(oldProperty.isRequired, newProperty.isRequired)
    }
    if (oldProperty.getSize != newProperty.getSize) {
      attributeChanges += SizeChange(oldProperty.getSize, newProperty.getSize)
    }
    if (oldProperty.getKind != newProperty.getKind) {
      attributeChanges += KindChange(oldProperty.getKind, newProperty.getKind)
    }
    if (!areEnumValuesEqual(oldProperty.getEnumValues, newProperty.getEnumValues)) {
      attributeChanges += EnumValuesChange(oldProperty.getEnumValues.asScala.toSeq, newProperty.getEnumValues.asScala.toSeq)
    }
    if (oldProperty.getReferencedType != newProperty.getReferencedType) {
      attributeChanges += ReferencedTypeChange(oldProperty.getReferencedType, newProperty.getReferencedType)
    }
    if (kindsWithDefaultValueConverter.contains(oldProperty.getKind) && kindsWithDefaultValueConverter.contains(newProperty.getKind)) {
      val oldDefaultValue = oldProperty.getDefaultValue()
      val newDefaultValue = newProperty.getDefaultValue()
      if (oldDefaultValue != newDefaultValue) {
        attributeChanges += DefaultChange(oldDefaultValue, newDefaultValue)
      }
    }
    if (oldProperty.isHidden != newProperty.isHidden) {
      attributeChanges += HiddenChange(oldProperty.isHidden, newProperty.isHidden)
    }
    if (oldProperty.isTransient != newProperty.isTransient) {
      attributeChanges += TransientChange(oldProperty.isTransient, newProperty.isTransient)
    }
    if (oldProperty.isReadonly != newProperty.isReadonly) {
      attributeChanges += ReadOnlyChange(oldProperty.isReadonly, newProperty.isReadonly)
    }
    if (oldProperty.getCandidateValuesFilter != newProperty.getCandidateValuesFilter) {
      attributeChanges += CandidateValuesFilter(oldProperty.getCandidateValuesFilter, newProperty.getCandidateValuesFilter)
    }
    if (oldProperty.getAliases != newProperty.getAliases) {
      attributeChanges += AliasesChange(oldProperty.getAliases.asScala.toSeq, newProperty.getAliases.asScala.toSeq)
    }
    if (notEqual(oldProperty.getInputHint, newProperty.getInputHint)) {
       attributeChanges += InputHintChange(oldProperty.getInputHint, newProperty.getInputHint)
    }
    if (oldProperty.getAnnotations != newProperty.getAnnotations) {
      attributeChanges += AnnotationsChange(oldProperty.getAnnotations.asScala.toSeq, newProperty.getAnnotations.asScala.toSeq)
    }
    if (oldProperty.isInspectionProperty != newProperty.isInspectionProperty) {
      attributeChanges += InspectionPropertyChange(oldProperty.isInspectionProperty, newProperty.isInspectionProperty)
    }
    if (attributeChanges.nonEmpty) {
      Some(UpdatedProperty(propertyName, attributeChanges = attributeChanges.toSeq))
    } else {
      None
    }
  }

  def areEnumValuesEqual(oldValues: java.util.List[String], newValues: java.util.List[String]): Boolean = {
    val o = Option(oldValues).map(_.asScala.toSet)
    val n = Option(newValues).map(_.asScala.toSet)

    o == n
  }
}
