package com.xebialabs.xlrelease.booter

import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.model.{DbPlugin, FilePlugin}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository.UpdatePluginStatus
import com.xebialabs.plugin.manager.rest.api.PluginStatus
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import com.xebialabs.xlrelease.events.system.distributed.ReloadPlugin
import com.xebialabs.xlrelease.events.system.{PluginReloaded, TypesReloaded}
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventBus}
import com.xebialabs.xlrelease.plugin.classloading.XlrPluginClassLoader
import grizzled.slf4j.Logging
import org.springframework.util.{FileCopyUtils, FileSystemUtils}

import java.io.File
import java.nio.file.{Files, Path}
import scala.jdk.StreamConverters._
import scala.util.Using

class XlrPluginEventHandler(eventBus: EventBus,
                            pluginsDir: Path,
                            pluginRepository: SqlPluginRepository,
                            xlrBooter: XlrBooter
                           ) extends Logging {

  @AsyncSubscribe
  def onEvent(event: ReloadPlugin): Unit = {
    val pluginId = event.pluginId
    val tccl = Thread.currentThread().getContextClassLoader
    val pluginClassLoader = tccl.asInstanceOf[XlrPluginClassLoader]

    val pluginFile = pluginClassLoader.withWriteLock {
      pluginClassLoader.closePlugin(pluginId)
      syncPlugin(pluginId) // this one is product specific
    }

    pluginClassLoader.reload(pluginId, pluginFile)

    eventBus.publish(PluginReloaded(pluginId))

    xlrBooter.reboot()

    eventBus.publish(TypesReloaded())
  }

  private def syncPlugin(pluginId: PluginId): File = {
    removePreviousVersions(pluginId)

    pluginRepository.getByNameAndVersion(pluginId.name, pluginId.version) match {
      case matches if matches.size == 1 =>
        writeLocalFile(matches.head, pluginId)
      case matches if matches.size > 1 =>
        throw new IllegalStateException(s"Unable to install plugin $pluginId: too many plugins matching name and version")
      case _ =>
        throw new IllegalStateException(s"Unable to install plugin $pluginId: plugin not found in the database")
    }
  }

  private def removePreviousVersions(pluginId: PluginId): Unit = {
    val plugins = Using.resource(Files.walk(pluginsDir)) { files =>
      files.filter(_.isFile)
        .filter(file => ConfigWrapper.isValidExtension(file.toFile.getName))
        .toScala(List)
        .map(FilePlugin(_))
        .filter(_.name == pluginId.name)
    }


    plugins.foreach { plugin =>
      logger.warn(s"Deleting previously installed plugin: ${plugin.filePath}")
      try {
        FileSystemUtils.deleteRecursively(plugin.filePath.toFile)
      } catch {
        case ex: Exception =>
          logger.error(s"Unable to delete ${plugin.name}", ex)
          throw ex
      }
    }
  }

  private def writeLocalFile(plugin: DbPlugin, pluginId: PluginId): File = {
    plugin.bytes match {
      case Some(pluginBytes) =>
        val targetFile = new File(pluginId.pluginFilePath(pluginsDir))
        FileCopyUtils.copy(pluginBytes, targetFile)
        pluginRepository.update(UpdatePluginStatus(plugin.name, PluginStatus.INSTALLED))
        targetFile
      case None => throw new IllegalStateException(s"Unable to install plugin $pluginId: no content found in the database")
    }
  }

}
