package com.xebialabs.xlrelease.plugin.manager.validator

import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, PropertyDescriptor}
import org.springframework.util.StringUtils

class ZipTypeChangeValidator extends TypeChangeValidator {

  override def validate(changes: Seq[Change]): TypeChangeValidationResults = {
    changes.foldLeft(TypeChangeValidationResults.empty()) { case (results, change) =>
      change match {
        case n: NewType =>
          results ++ TypeChangeValidationResults.empty()
        case d: DeletedType =>
          // Trying to determine if a type is in use anywhere in the system is expensive/impossible?
          //  - Are any types extending it in their type definition?
          //  - Are any types referencing it in their type definition?
          //  - If it is a task does any template or live/archived release contain a task of that type
          results ++ error(formatMessage(TypeContext(d), "can't be deleted"))
        case u: UpdatedType =>
          results ++ validateUpdatedType(u)
        case _ => results
      }
    }
  }

  private def validateUpdatedType(change: UpdatedType): TypeChangeValidationResults = {
    val typeContext = TypeContext(change)

    val attributeChanges = change.attributeChanges.foldLeft(TypeChangeValidationResults.empty()) { case (results, attributeChange) =>
      results ++ validateTypeAttributeChanges(typeContext, attributeChange)
    }

    change.propertyChanges.foldLeft(attributeChanges) { case (results, propertyChange) =>
      propertyChange match {
        // New properties should be fine
        case n: NewProperty => results
        case d: DeletedProperty =>
          // TODO: S-92953 would allow deleted properties and this would change so it is no longer an error, maybe leave as a warning?
          val propertyContext = PropertyContext(typeContext, d)
          results ++ error(formatMessage(propertyContext, "can't be deleted"))
        case u: UpdatedProperty => results ++ validateUpdatedProperty(typeContext, u)
      }
    }
  }

  private def validateTypeAttributeChanges(typeContext: TypeContext, attributeChange: AttributeChange[_]): TypeChangeValidationResults = {
    attributeChange match {
      case VirtualChange(false, true) => error(formatMessage(typeContext, attributeChange.toString))
      case VirtualChange(true, false) => TypeChangeValidationResults.empty()
      case VersionedChange(_, _) => TypeChangeValidationResults.empty()
      case _ =>
        // There may be some scenarios where changes to other type attributes would be ok
        //  - superclass/interfaces changes might be safe as long as the change is only adding another class to the
        //    hierarchy.  Determining if/when it is safe seems complicated though so for now just always fail
        error(formatMessage(typeContext, attributeChange.toString))
    }
  }


  private def validateUpdatedProperty(typeContext: TypeContext, change: UpdatedProperty): TypeChangeValidationResults = {
    val propertyContext = PropertyContext(typeContext, change)
    change.attributeChanges
      .filterNot(_.cosmetic) // Cosmetic changes only impact presentation of types in UI so should always be safe
      .foldLeft(TypeChangeValidationResults.empty()) { case (results, attributeChange) =>
        results ++ validatePropertyAttributeChanges(propertyContext, attributeChange)
      }
  }

  private def validatePropertyAttributeChanges(propertyContext: PropertyContext, attributeChange: AttributeChange[_]): TypeChangeValidationResults = {
    attributeChange match {
      case _: AsContainmentChange => error(formatMessage(propertyContext, attributeChange.toString))
      case _: NestedChange => error(formatMessage(propertyContext, attributeChange.toString))
      case _: PasswordChange => error(formatMessage(propertyContext, attributeChange.toString))
      case required: RequiredChange => validateRequiredAttributeChange(propertyContext, required)
      case _: KindChange => error(formatMessage(propertyContext, attributeChange.toString))
      case enumValues: EnumValuesChange => validateEnumValuesChange(propertyContext, enumValues)
      case _: ReferencedTypeChange => error(formatMessage(propertyContext, attributeChange.toString))
      case defaultValue: DefaultChange => validateDefaultValueChange(propertyContext, defaultValue)
      case _: HiddenChange => error(formatMessage(propertyContext, attributeChange.toString))
      case _: TransientChange => error(formatMessage(propertyContext, attributeChange.toString))
      case _: ReadOnlyChange => warn(formatMessage(propertyContext, attributeChange.toString))
      case aliases: AliasesChange => validateAliasChange(propertyContext, aliases)
      case _: AnnotationsChange => error(formatMessage(propertyContext, attributeChange.toString))
      case _ =>
        warn(attributeChange.toString)
    }
  }

  private def validateRequiredAttributeChange(propertyContext: PropertyContext, change: RequiredChange): TypeChangeValidationResults = {
    if (change.newValue) {
      val defaultValue = propertyContext.getNewPropertyDescriptor().get.getDefaultValue
      val hasDefault = defaultValue match {
        case null => false
        case s: String => StringUtils.hasText(s)
        case _ => true
      }

      if (!hasDefault) {
        warn(formatMessage(propertyContext, change.toString)) ++
          error(formatMessage(propertyContext, "changing the property to required must also have a defaultValue defined"))
      }
      else {
        TypeChangeValidationResults.empty()
      }
    } else {
      TypeChangeValidationResults.empty()
    }
  }

  private def validateDefaultValueChange(propertyContext: PropertyContext, change: DefaultChange): TypeChangeValidationResults = {
    val hasDefault = change.newValue match {
      case null => false
      case s: String => s != null && StringUtils.hasText(s)
      case o: Object => o != null
    }

    val isRequired = propertyContext.getNewPropertyDescriptor().get.isRequired()

    if (isRequired && !hasDefault) {
      error(formatMessage(propertyContext, change.toString)) ++
        error(formatMessage(propertyContext, "removing the defaultValue is not allowed when the property is required"))
    } else {
      TypeChangeValidationResults.empty()
    }
  }

  private def validateEnumValuesChange(propertyContext: PropertyContext, change: EnumValuesChange): TypeChangeValidationResults = {
    // it is perfectly valid to restrict enum values, see com.xebialabs.deployit.booter.local.LocalPropertyDescriptor.initEnumValues:172
    val isSubset = change.newValue.toSet.subsetOf(change.oldValue.toSet)
    if (isSubset) {
      TypeChangeValidationResults.empty()
    } else {
      val removedValues = change.oldValue diff change.newValue
      if (removedValues.nonEmpty) {
        error(formatMessage(propertyContext, change.toString)) ++
          error(formatMessage(propertyContext, s"values [${removedValues.mkString(",")}] can't be removed"))
      } else {
        TypeChangeValidationResults.empty()
      }
    }
  }

  private def validateAliasChange(propertyContext: PropertyContext, change: AliasesChange): TypeChangeValidationResults = {
    val removedValues = change.oldValue diff change.newValue

    if (removedValues.nonEmpty) {
      error(formatMessage(propertyContext, change.toString)) ++
        error(formatMessage(propertyContext, s"values [${removedValues.mkString(",")}] can't be removed"))
    }
    else {
      TypeChangeValidationResults.empty()
    }
  }

  private def warn(message: String): TypeChangeValidationResults = {
    TypeChangeValidationResults(Seq(message), Seq.empty)
  }

  private def error(message: String): TypeChangeValidationResults = {
    TypeChangeValidationResults(Seq.empty, Seq(message))
  }

  sealed trait ValidationContext

  private case class TypeContext(typeChange: TypeChange) extends ValidationContext {
    override def toString: String = s"Type [${typeChange.xlTypeName}]"

    // this will always be a property descriptor from old type
    def getOldDescriptor(): Option[Descriptor] = {
      typeChange.oldXlType.map(_.getDescriptor)
    }

    def getNewDescriptor(): Option[Descriptor] = {
      typeChange.newXlType.map(_.getDescriptor)
    }

  }

  private case class PropertyContext(typeContext: TypeContext, propertyChange: PropertyChange) extends ValidationContext {
    override def toString: String = s"${typeContext}, property [${propertyChange.propertyName}]"

    // this will always be a property descriptor from old type
    def getOldPropertyDescriptor(): Option[PropertyDescriptor] = {
      typeContext.getOldDescriptor().map(_.getPropertyDescriptor(propertyChange.propertyName))
    }

    def getNewPropertyDescriptor(): Option[PropertyDescriptor] = {
      typeContext.getNewDescriptor().map(_.getPropertyDescriptor(propertyChange.propertyName))
    }

  }

  private def formatMessage(validationContext: ValidationContext, message: String): String = {
    s"${validationContext}: ${message}"
  }
}
