package com.xebialabs.xlrelease.api.v1;

import java.util.List;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.form.CreateVersionForm;
import com.xebialabs.xlrelease.api.v1.views.VersionInfo;
import com.xebialabs.xlrelease.api.v1.views.VersionsView;
import com.xebialabs.xlrelease.domain.versioning.ascode.GitBranch;
import com.xebialabs.xlrelease.domain.versioning.ascode.VersioningStyle;
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings;
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationReport;
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationReportMessage;

import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

/**
 * Operations to store and synchronize folder contents - templates, workflows,
 * connections, variables, and more - as YAML files in Git for version tracking.
 */
@Path("api/v1/version-control")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
@PublicApi
@ShowOnlyPublicApiMembers
public interface FolderVersioningApi extends ApiService {

    default String serviceName() {
        return "folderVersioningApi";
    }

    /**
     * Retrieves the Git versioning settings for the specified folder.
     *
     * @param folderId the unique identifier of the folder whose versioning settings are to be retrieved.
     * @return the current versioning settings associated with the folder.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/")
    @PublicApiMember
    FolderVersioningSettings getSettings(@PathParam("folderId") String folderId);

    /**
     * Updates the Git versioning settings for the specified folder.
     *
     * @param folderId the unique identifier of the folder to update versioning settings for.
     * @param config   the new versioning configuration to apply.
     * @return the updated versioning settings for the folder.
     */
    @PUT
    @Path("/{folderId:.*Folder[^/-]*}/")
    @PublicApiMember
    FolderVersioningSettings updateSettings(@PathParam("folderId") String folderId, FolderVersioningSettings config);

    /**
     * Deletes the Git versioning settings and versioned history for the specified folder.
     * This does not affect existing data in the Git repository.
     *
     * @param folderId the unique identifier of the folder whose versioning settings should be deleted.
     */
    @DELETE
    @Path("/{folderId:.*Folder[^/-]*}/")
    @PublicApiMember
    void deleteSettings(@PathParam("folderId") String folderId);

    /**
     * Retrieves all versions associated with the specified folder.
     *
     * @param folderId     the unique identifier of the folder whose versions are to be retrieved.
     * @param fetchChanges whether to fetch changes from the repository before returning versions.
     * @return the list of versions for the specified folder.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/versions")
    @PublicApiMember
    VersionsView getVersions(@PathParam("folderId") String folderId, @DefaultValue("false") @QueryParam("fetchChanges") Boolean fetchChanges);

    /**
     * Retrieves all versions associated with the specified folder. Equivalent to calling getVersions(folderId, false).
     *
     * @param folderId the unique identifier of the folder whose versions are to be retrieved.
     * @return the list of versions for the specified folder.
     */
    @PublicApiMember
    VersionsView getVersions(String folderId);

    /**
     * Retrieves all branches associated with the specified folder.
     *
     * @param folderId the unique identifier of the folder whose branches are to be retrieved.
     * @param configId the configuration ID to filter branches (optional).
     * @return a list of branches for the specified folder.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/branches")
    List<GitBranch> getAllBranches(@PathParam("folderId") String folderId, @QueryParam("configId") String configId);

    @POST
    @Path("/{folderId:.*Folder[^/-]*}/versions")
    VersionInfo createVersion(@PathParam("folderId") String folderId, CreateVersionForm versionForm);

    /**
     * Creates a new version for the specified folder using the provided version name and description.
     *
     * @param folderId    the unique identifier of the folder for which the version is to be created.
     * @param versionName the name to assign to the new version; must be non-null and unique within the folder.
     * @param description a brief description of the version (maximum 256 characters).
     * @return information about the newly created version.
     */
    @PublicApiMember
    VersionInfo createVersion(String folderId, String versionName, String description);

    /**
     * Applies the specified version to the given folder.
     *
     * @param folderId the unique identifier of the folder to which the version will be applied.
     * @param version  the version to apply to the folder.
     * @return information regarding the application of the version, including any warnings or errors encountered during the process.
     */
    @POST
    @Path("/{folderId:.*Folder[^/-]*}/versions/{version:.*}/apply")
    @PublicApiMember
    ValidationReport applyVersion(@PathParam("folderId") String folderId, @PathParam("version") String version);

    /**
     * Returns a YAML preview of the current state of the specified folder.
     *
     * @param folderId the id of the folder to preview.
     * @param filename the name of the file to preview (optional).
     * @return a YAML string representing the current state of the folder.
     */
    @GET
    @Produces("text/vnd.yaml")
    @Path("/{folderId:.*Folder[^/-]*}/preview")
    String previewCurrent(@PathParam("folderId") String folderId, @QueryParam("filename") String filename);

    /**
     * Returns a YAML preview of the specified version of the folder.
     *
     * @param folderId the id of the folder to preview
     * @param version  the version identifier to preview
     * @param filename the name of the file to preview (optional)
     * @return a YAML string representing the specified version of the folder
     */
    @GET
    @Produces("text/vnd.yaml")
    @Path("/{folderId:.*Folder[^/-]*}/versions/{version:.*}/preview")
    String previewVersion(@PathParam("folderId") String folderId, @PathParam("version") String version, @QueryParam("filename") String filename);

    /**
     * Returns the list of file names that are associated with the specified version of a folder.
     *
     * @param folderId the unique identifier of the folder for which to retrieve versioned file names.
     * @param version  the version of the folder for which to retrieve file names.
     * @return a list of file names present in the specified version of the folder.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/versions/{version:.*}/getVersionedFileNames")
    @PublicApiMember
    List<String> getVersionedFileNames(@PathParam("folderId") String folderId, @PathParam("version") String version);

    /**
     * Retrieves the list of file names that are currently versionable in the specified folder given the versioning settings for the folder.
     *
     * @param folderId the unique identifier of the folder for which to retrieve versionable file names.
     * @return a list of file names that are currently versionable in the folder.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/getVersionableFileNames")
    @PublicApiMember
    List<String> getCurrentVersionableFileNames(@PathParam("folderId") String folderId);

    @DELETE
    @Path("/{folderId:.*Folder[^/-]*}/delete-local-repo")
    void deleteLocalRepo(@PathParam("folderId") String folderId, @DefaultValue("false") @QueryParam("clusterWide") Boolean clusterWide);

    void deleteLocalRepo(String folderId);

    @PUT
    @Path("/{folderId:.*Folder[^/-]*}/reset-local-repo")
    void resetLocalRepo(@PathParam("folderId") String folderId, @DefaultValue("false") @QueryParam("clusterWide") Boolean clusterWide);

    /**
     * Validates the current state of the specified folder to identify issues with password or unreferenced variables that would be present if the folder were
     * versioned.
     *
     * @param folderId the unique identifier of the folder to validate.
     * @return information detailing the results of the validation including any warnings or errors that would occur if the folder were versioned.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/validate")
    @PublicApiMember
    ValidationReport validateCurrent(@PathParam("folderId") String folderId);

    /**
     * Retrieves the validation messages for the currently applied version for the specified folder.
     *
     * @param folderId the unique identifier of the folder to retrieve validation messages for.
     * @return information detailing the validation messages that are present for the folder.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/validation-messages")
    @PublicApiMember
    ValidationReport getValidationMessages(@PathParam("folderId") String folderId);

    /**
     * Sets the validation messages for the specified folder.
     *
     * @param folderId         the unique identifier of the folder for which to set validation messages.
     * @param validationReport validation messages to associate with the folder.
     */
    @PUT
    @Path("/{folderId:.*Folder[^/-]*}/validation-messages")
    void setValidationMessages(@PathParam("folderId") String folderId, ValidationReport validationReport);

    /**
     * Retrieves a specific validation message for the given folder and message ID for passwords or unversioned references.
     *
     * @param folderId  the unique identifier of the folder containing the validation message.
     * @param messageId the unique identifier of the validation message to retrieve.
     * @return the validation report message corresponding to the specified message ID in the folder
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/message/{messageId:.*}/")
    @PublicApiMember
    ValidationReportMessage getValidationMessage(@PathParam("folderId") String folderId, @PathParam("messageId") String messageId);

    /**
     * Updates the status of a specific validation message for the given folder and message ID.
     *
     * @param folderId          the unique identifier of the folder containing the validation message.
     * @param messageId         the unique identifier of the validation message to update.
     * @param validationMessage the updated validation report message
     */
    @PUT
    @Path("/{folderId:.*Folder[^/-]*}/message/{messageId:.*}/update")
    void setValidationMessagesStatus(@PathParam("folderId") String folderId, @PathParam("messageId") String messageId, ValidationReportMessage validationMessage);

    @DELETE
    @Path("/{folderId:.*Folder[^/-]*}/clear-secrets")
    void clearSecrets(@PathParam("folderId") String folderId);

    /**
     * Retrieves the list of available versioning styles.
     *
     * @return a list of supported versioning styles
     */
    @GET
    @Path("/styles")
    List<VersioningStyle> getVersioningStyles();

    /**
     * Compares two versions of a specified file within a given folder and returns the results highlighting the differences.
     * If either version parameter is omitted, the current version of the file will be used for that side of the comparison.
     *
     * @param folderId the unique identifier of the folder that contains the file whose versions are to be compared.
     * @param fileName the name of the file to compare.
     * @param leftVersion the version identifier of the source (left) file for comparison. If not specified, the current version is used.
     * @param rightVersion the version identifier of the target (right) file for comparison. If not specified, the current version is used.
     * @return a String containing the comparison results that identify differences between the specified versions.
     */
    @GET
    @Path("/{folderId:.*Folder[^/-]*}/compare")
    @PublicApiMember
    String compareVersions(@PathParam("folderId") String folderId, @QueryParam("fileName") String fileName,
                               @QueryParam("leftVersion") String leftVersion, @QueryParam("rightVersion") String rightVersion);

    /**
     * Compares a specified version of a file with its latest version in the given folder and returns the results highlighting the differences.
     *
     * @param folderId the unique identifier of the folder containing the file to be compared.
     * @param fileName the name of the file to compare.
     * @param version the version identifier of the file to compare against the latest version.
     * @return a String containing the comparison results identifying the differences between the specified version and the latest version.
     */
    @PublicApiMember
    String compareWithLatest(@PathParam("folderId") String folderId, @QueryParam("fileName") String fileName, @QueryParam("version") String version);

}
