package com.xebialabs.xlrelease.api.v1;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.CreateTeam;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.ReleaseKind;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.rest.AllCILevels;

/**
 * Operations on folders.
 */
@Path("/api/v1/folders")
@Produces({MediaType.APPLICATION_JSON})
@Consumes({MediaType.APPLICATION_JSON})
@AllCILevels
@PublicApi
@ShowOnlyPublicApiMembers
public interface FolderApi extends ApiService {

    String EXPRESSION_FOLDER_ID = "/{folderId:.*(Folder[^/]*|Applications)}";

    default String serviceName() {
        return "folderApi";
    }

    /**
     * Returns a list of folders from the root directory.
     *
     * @param page                    the page of results to return. Defaults at 0 if page set to null
     * @param resultPerPage           the number of results per page. Defaults at 50 if set to null
     * @param depth                   the depth to search for. Defaults at 1 if set to null
     * @param decorateWithPermissions the boolean to decorate the folders with the effective permissions. Defaults to false if set to null
     * @return a list of the folders
     */
    @GET
    @Path("/list")
    @PublicApiMember
    List<Folder> listRoot(@QueryParam("page") Long page,
                          @QueryParam("resultsPerPage") Long resultPerPage,
                          @QueryParam("depth") Integer depth,
                          @QueryParam("permissions") Boolean decorateWithPermissions);

    /**
     * Returns a list of all folders directly under the root directory.
     *
     * @return a list of the folders
     */
    @PublicApiMember
    default List<Folder> listRoot() {
        return listRoot(null, null, null, null);
    }

    /**
     * Lists the folders inside a given folder.
     *
     * @param parentId      the parent folder to retrieve from
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @param depth         the depth to search for. Defaults at 1
     * @return the list of folders beneath a given folder
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/list")
    @PublicApiMember
    List<Folder> list(@PathParam("folderId") String parentId,
                      @QueryParam("page") Long page,
                      @QueryParam("resultsPerPage") Long resultPerPage,
                      @QueryParam("depth") Integer depth,
                      @QueryParam("permissions") Boolean decorateWithPermissions);

    /**
     * Finds a folder from a given path.
     *
     * @param path  the path for the folder to search on
     * @param depth the depth to search for. Defaults at 1
     * @return the folder found in the path
     */
    @GET
    @Path("/find")
    @PublicApiMember
    Folder find(@QueryParam("byPath") String path, @QueryParam("depth") Integer depth);

    /**
     * Returns the specified folder.
     *
     * @param folderId the id of the folder
     * @param depth    the depth to search for. Defaults at 1
     * @return the specified folder
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID)
    @PublicApiMember
    Folder getFolder(@PathParam("folderId") String folderId, @QueryParam("depth") Integer depth);

    /**
     * Returns the specified folder with default depth.
     *
     * @param folderId the folder id
     * @return the given folder
     */
    @PublicApiMember
    Folder getFolder(String folderId);

    /**
     * Returns the list of release or workflow templates for the given folder.
     *
     * @param folderId      the id of the folder
     * @param kind          the kind of template. Default value is release.
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @param depth         the (optional) depth to search for; defaults at 1
     * @return the list of templates in the folder
     * @deprecated Since 24.1.0. Please use {@link #getTemplates(String, String, List, String, Long, Long)}
     */
    @Deprecated(since = "24.1.0")
    @PublicApiMember
    List<Release> getTemplates(String folderId, String kind, Long page, Long resultPerPage, Integer depth);

    /**
     * Returns the list of release or workflow templates for the given folder.
     *
     * @param folderId      the id of the folder
     * @param title         an optional search filter containing the title of the template.
     * @param tags          an optional search filter containing list of template tags.
     * @param kind          the kind of template. Default value is release.
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @return the list of templates in the folder
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/templates")
    @PublicApiMember
    List<Release> getTemplates(@PathParam("folderId") String folderId,
                               @QueryParam("title") String title,
                               @QueryParam("tag") List<String> tags,
                               @DefaultValue("RELEASE") @QueryParam("kind") String kind,
                               @QueryParam("page") Long page,
                               @QueryParam("resultsPerPage") Long resultPerPage);

    /**
     * Returns the templates for the given folder.
     *
     * @param folderId      the id of the folder
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @param depth         the (optional) depth to search for; defaults at 1
     * @return the list of templates in the folder
     * @deprecated Since 24.1.0. Please use {@link #getTemplates(String, String, List, Long, Long)}
     */
    @Deprecated(since = "24.1.0")
    @PublicApiMember
    default List<Release> getTemplates(String folderId, Long page, Long resultPerPage, Integer depth) {
        return getTemplates(folderId, ReleaseKind.RELEASE.name(), page, resultPerPage, depth);
    }

    /**
     * Returns the templates for the given folder.
     *
     * @param folderId      the id of the folder
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @return the list of templates in the folder
     */
    @PublicApiMember
    default List<Release> getTemplates(String folderId, Long page, Long resultPerPage) {
        return getTemplates(folderId, null, null, ReleaseKind.RELEASE.name(), page, resultPerPage);
    }

    /**
     * Returns the templates for the given folder.
     *
     * @param folderId      the id of the folder
     * @param title          an optional search filter containing the title of the template.
     * @param tags           an optional search filter containing list of template tags.
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @return the list of templates in the folder
     */
    @PublicApiMember
    default List<Release> getTemplates(String folderId, String title, List<String> tags, Long page, Long resultPerPage) {
        return getTemplates(folderId, title, tags, ReleaseKind.RELEASE.name(), page, resultPerPage);
    }

    /**
     * Returns the templates for the given folder.
     *
     * @param folderId      the id of the folder
     * @param kind          the kind of template. Default value is release.
     * @param page          the page of results to return. Defaults at 0
     * @param resultPerPage the number of results per page. Defaults at 50
     * @return the list of templates in the folder
     */
    @PublicApiMember
    List<Release> getTemplates(String folderId, String kind, Long page, Long resultPerPage);

    /**
     * Returns the templates for the given folder with default pagination and depth.
     *
     * @param folderId the folder id
     * @return the list of templates in the folder
     */
    @PublicApiMember
    List<Release> getTemplates(String folderId);

    /**
     * Returns the releases started from this folder.
     *
     * @param folderId        the id of the folder
     * @param page            the page of results to return. Defaults at 0
     * @param resultPerPage   the number of results per page. Defaults at 50
     * @param depth           the depth to search for
     * @param releasesFilters releaseFilter to filter the release search
     * @return the list of releases
     * @deprecated Since 24.1.0. Please use {@link #searchReleases(String, Long, Long, ReleasesFilters)}
     */
    @Deprecated(since = "24.1.0")
    @POST
    @Path(EXPRESSION_FOLDER_ID + "/releases")
    @PublicApiMember
    List<Release> searchReleases(@PathParam("folderId") String folderId,
                                 @QueryParam("page") Long page,
                                 @QueryParam("numberbypage") Long resultPerPage,
                                 @QueryParam("depth") Integer depth,
                                 ReleasesFilters releasesFilters);

    /**
     * Returns the releases started from this folder.
     *
     * @param folderId        the id of the folder
     * @param page            the page of results to return. Defaults at 0
     * @param resultPerPage   the number of results per page. Defaults at 50
     * @param releasesFilters releaseFilter to filter the release search
     * @return the list of releases
     */
    @POST
    @Path(EXPRESSION_FOLDER_ID + "/releases")
    @PublicApiMember
    List<Release> searchReleases(@PathParam("folderId") String folderId,
                                 @QueryParam("page") Long page,
                                 @QueryParam("numberbypage") Long resultPerPage,
                                 ReleasesFilters releasesFilters);

    /**
     * Returns the releases started from this folder with default pagination and depth.
     *
     * @param folderId        the folder id
     * @param releasesFilters the releaseFilters to filter the release search on
     * @return the list of releases in a folder
     */
    @PublicApiMember
    List<Release> searchReleases(String folderId, ReleasesFilters releasesFilters);

    /**
     * Move a template to another folder.
     *
     * @param folderId   the target folder the template will be moved to
     * @param templateId the id of the template to be moved
     */
    @POST
    @Path(EXPRESSION_FOLDER_ID + "/templates/{templateId:.*Release[^/]*}")
    @PublicApiMember
    void moveTemplate(@PathParam("folderId") String folderId,
                      @PathParam("templateId") String templateId,
                      @QueryParam("mergePermissions") Boolean mergeTeams);


    /**
     * Move a template to another folder without merging permissions.
     *
     * @param folderId   the id of the parent folder
     * @param templateId the id of the template to be moved
     */
    @PublicApiMember
    void moveTemplate(String folderId, String templateId);

    /**
     * Adds a new folder inside the specified folder.
     *
     * @param parentId the id of the folder to create the folder in
     * @param folder   the folder to create the new folder in
     * @return the newly created folder
     */
    @POST
    @Path(EXPRESSION_FOLDER_ID)
    @PublicApiMember
    Folder addFolder(@PathParam("folderId") String parentId, Folder folder);

    /**
     * Deletes the specified folder and all the contents inside.
     *
     * @param folderId the id of the folder to delete
     */
    @DELETE
    @Path(EXPRESSION_FOLDER_ID)
    @PublicApiMember
    void delete(@PathParam("folderId") String folderId);

    /**
     * Move a folder to another folder. Following restrictions apply to folder being moved: it must not contain any scheduled or running releases, the folder
     * and all its children must not contain any triggers, and it must not contain any releases with references to connections not available under new parent folder.
     *
     * @param folderId    the id of the folder to move
     * @param newParentId the id of the parent folder the folder will be moved under
     * @return the moved folder
     */
    @POST
    @Path(EXPRESSION_FOLDER_ID + "/move")
    @PublicApiMember
    Folder move(@PathParam("folderId") String folderId, @QueryParam("newParentId") String newParentId);

    /**
     * Rename a folder.
     *
     * @param folderId the id of the folder to rename
     * @param newName  the new name of the folder
     */
    @POST
    @Path(EXPRESSION_FOLDER_ID + "/rename")
    @PublicApiMember
    void rename(@PathParam("folderId") String folderId, @QueryParam("newName") String newName);

    /**
     * Returns possible permissions.
     *
     * @return list of permissions for folders, templates and releases
     */
    @GET
    @Path("/permissions")
    @PublicApiMember
    List<String> getPermissions();

    //TODO: figure out a way to deal with TeamView

    /**
     * Returns folder effective teams.
     *
     * @param folderId the id of the folder
     * @return list of effective teams
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/teams")
    @PublicApiMember
    List<TeamView> getTeams(@PathParam("folderId") String folderId);

    /**
     * Set teams of the folder.
     * <p>
     * If teams were previously inherited they will be detached after the update.
     * To use inherited teams on a folder submit empty list of teams.
     *
     * @param folderId the identifier of the folder
     * @return a list of updated teams
     */
    @POST
    @Path(EXPRESSION_FOLDER_ID + "/teams")
    @PublicApiMember
    List<TeamView> setTeams(@PathParam("folderId") String folderId, List<TeamView> teamDtos);

    @GET
    @Path(EXPRESSION_FOLDER_ID + "/folderOwner")
    boolean isFolderOwner(@PathParam("folderId") String folderId);

    /**
     * Returns folder variables list.
     *
     * @param folderId   the folder identifier.
     * @param folderOnly filter if you only want to get information from your parentId. (optional)
     * @return variables list.
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/variables")
    @PublicApiMember
    List<Variable> listVariables(@PathParam("folderId") String folderId, @QueryParam("folderOnly") boolean folderOnly);

    /**
     * Returns folder variables list with replaced variable values.
     *
     * @param folderId   the folder identifier.
     * @param folderOnly filter if you only want to get information from your parentId. (optional)
     * @return variables list.
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/variableValues")
    @PublicApiMember
    Map<String, String> listVariableValues(@PathParam("folderId") String folderId, @QueryParam("folderOnly") boolean folderOnly);

    /**
     * Adds a folder variable.
     *
     * @param folderId the folder identifier.
     * @param variable the folder variable to add.
     * @return the added folder variable.
     */
    @POST
    @PublicApiMember
    @Path(EXPRESSION_FOLDER_ID + "/variables")
    Variable createVariable(@PathParam("folderId") String folderId,
                            com.xebialabs.xlrelease.api.v1.forms.Variable variable);

    /**
     * Returns the folder variable for the given identifier.
     *
     * @param folderId   the folder identifier.
     * @param variableId the variable identifier.
     * @return the variable.
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/{variableId:.*Variable[^/]*}")
    @PublicApiMember
    Variable getVariable(@PathParam("folderId") String folderId, @PathParam("variableId") String variableId);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{variableId:.*Variable[^/]*}")
    Variable updateVariable(@PathParam("folderId") String folderId, @PathParam("variableId") String variableId, Variable variable);

    /**
     * Updates properties of a folder variable by its ID.
     *
     * @param folderId the folder identifier.
     * @param variable the variable to update.
     * @return the updated variable.
     */
    @PublicApiMember
    Variable updateVariable(String folderId, Variable variable);

    /**
     * Deletes a folder variable.
     *
     * @param folderId   the folder identifier.
     * @param variableId the variable identifier.
     */
    @DELETE
    @Path(EXPRESSION_FOLDER_ID + "/{variableId:.*Variable[^/]*}")
    @PublicApiMember
    void deleteVariable(@PathParam("folderId") String folderId, @PathParam("variableId") String variableId);

    /**
     * Returns possible values for the variable with the given identifier.
     *
     * @param variableId the variable identifier.
     * @return possible values for the variable.
     */
    @GET
    @Path(EXPRESSION_FOLDER_ID + "/{variableId:.*Variable[^/]*}/possibleValues")
    @PublicApiMember
    Collection<Object> getVariablePossibleValues(@PathParam("variableId") String variableId);

    @GET
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}")
    @PublicApiMember
    Team getTeam(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId);

    @GET
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}")
    @PublicApiMember
    Team findTeam(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName);

    @POST
    @Path(EXPRESSION_FOLDER_ID + "/create-team")
    String createTeam(@PathParam("folderId") String folderId, CreateTeam createTeam);

    @PublicApiMember
    default String createTeam(String folderId, String name, List<String> principals, List<String> roles, List<String> permissions) {
        return createTeam(folderId, new CreateTeam(name, principals, roles, permissions));
    }

    @DELETE
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}")
    @PublicApiMember
    void deleteTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName);

    @DELETE
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}")
    @PublicApiMember
    void deleteTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}/add-principals")
    @PublicApiMember
    void addPrincipalsToTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId, List<String> principalNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}/add-principals")
    @PublicApiMember
    void addPrincipalsToTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName, List<String> principalNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}/remove-principals")
    @PublicApiMember
    void removePrincipalsFromTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId, List<String> principalNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}/remove-principals")
    @PublicApiMember
    void removePrincipalsFromTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName, List<String> principalNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}/add-roles")
    @PublicApiMember
    void addRoleNamesToTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId, List<String> roleNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}/add-roles")
    @PublicApiMember
    void addRoleNamesToTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName, List<String> roleNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}/remove-roles")
    @PublicApiMember
    void removeRoleNamesFromTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId, List<String> roleNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}/remove-roles")
    @PublicApiMember
    void removeRoleNamesFromTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName, List<String> roleNames);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}/add-permissions")
    @PublicApiMember
    void addPermissionsToTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId, List<String> permissions);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}/add-permissions")
    @PublicApiMember
    void addPermissionsToTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName, List<String> permissions);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/{teamId:Team[^/]*}/remove-permissions")
    @PublicApiMember
    void removePermissionsFromTeamById(@PathParam("folderId") String folderId, @PathParam("teamId") String teamId, List<String> permissions);

    @PUT
    @Path(EXPRESSION_FOLDER_ID + "/teams/{teamName}/remove-permissions")
    @PublicApiMember
    void removePermissionsFromTeamByName(@PathParam("folderId") String folderId, @PathParam("teamName") String teamName, List<String> permissions);

    @PublicApiMember
    List<String> getPathSegments(String folderId);

    @PublicApiMember
    String getPath(String folderId);
}
