package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.xlrelease.api.v1.forms.TenantLinkRequest;
import com.xebialabs.xlrelease.api.v1.forms.TenantRequest;
import com.xebialabs.xlrelease.configuration.TenantLimit;
import com.xebialabs.xlrelease.domain.Tenant;

/**
 * REST API for managing tenants and their configurations in Release.
 */
@Path("/api/v1/tenants")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public interface TenantApi {

    /**
     * Retrieves all tenants in the system.
     *
     * @return a list of all tenants
     */
    @GET
    List<Tenant> getTenants();

    /**
     * Retrieves a specific tenant by its ID.
     *
     * @param tenantId the unique identifier of the tenant
     * @return the tenant with the specified ID
     */
    @GET
    @Path("/{tenantId}")
    Tenant getTenant(@PathParam("tenantId") String tenantId);

    /**
     * Creates a new tenant.
     *
     * @param tenantRequest the tenant creation request containing tenant details
     * @return the newly created tenant
     */
    @POST
    Tenant create(TenantRequest tenantRequest);

    /**
     * Updates an existing tenant.
     *
     * @param tenantId      the unique identifier of the tenant to update
     * @param tenantRequest the tenant update request containing new tenant details
     */
    @PUT
    @Path("/{tenantId}")
    void update(@PathParam("tenantId") String tenantId, TenantRequest tenantRequest);

    /**
     * Deletes a tenant by its ID.
     *
     * @param tenantId the unique identifier of the tenant to delete
     */
    @DELETE
    @Path("/{tenantId}")
    void delete(@PathParam("tenantId") String tenantId);

    /**
     * Retrieves all tenant limits in the system.
     *
     * @return a list of all tenant limits
     */
    @GET
    @Path("/limits")
    List<TenantLimit> getLimits();

    /**
     * Retrieves the limit configuration for a specific tenant.
     *
     * @param tenantId the unique identifier of the tenant
     * @return the tenant limit configuration
     */
    @GET
    @Path("/limits/{tenantId}")
    TenantLimit getLimit(@PathParam("tenantId") String tenantId);

    /**
     * Creates a new limit configuration for a tenant.
     *
     * @param tenantId the unique identifier of the tenant
     * @param limit    the tenant limit configuration to create
     * @return the newly created tenant limit configuration
     */
    @POST
    @Path("/limits/{tenantId}")
    TenantLimit createLimit(@PathParam("tenantId") String tenantId, TenantLimit limit);

    /**
     * Updates the limit configuration for a tenant.
     *
     * @param tenantId the unique identifier of the tenant
     * @param limit    the updated tenant limit configuration
     */
    @PUT
    @Path("/limits/{tenantId}")
    void updateLimit(@PathParam("tenantId") String tenantId, TenantLimit limit);

    /**
     * Deletes the limit configuration for a tenant.
     *
     * @param tenantId the unique identifier of the tenant
     */
    @DELETE
    @Path("/limits/{tenantId}")
    void deleteLimit(@PathParam("tenantId") String tenantId);

    /**
     * Links a tenant with various XL Release objects/resources.
     * This operation associates the tenant with different resources in the system.
     *
     * @param tenantId     the unique identifier of the tenant
     * @param linkRequests a list of tenant link requests specifying which resources to link
     */
    @PUT
    @Path("/{tenantId}/link")
    void linkResource(@PathParam("tenantId") String tenantId, List<TenantLinkRequest> linkRequests);
}
