package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.CompleteTransition;
import com.xebialabs.xlrelease.api.v1.forms.DeliveryFilters;
import com.xebialabs.xlrelease.api.v1.forms.DeliveryOrderMode;
import com.xebialabs.xlrelease.api.v1.views.DeliveryFlowReleaseInfo;
import com.xebialabs.xlrelease.api.v1.views.DeliveryTimeline;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.delivery.*;

/**
 * Operations on release deliveries.
 */
@Path("/api/v1/deliveries")
@Produces({MediaType.APPLICATION_JSON})
@Consumes({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface DeliveryApi extends ApiService {

    default String serviceName() {
        return "deliveryApi";
    }

    /**
     * Returns a delivery by ID.
     *
     * @param deliveryId the delivery identifier.
     * @return the delivery which has the given identifier.
     */
    @GET
    @Path("/{deliveryId:.*Delivery[^/]*}")
    @PublicApiMember
    Delivery getDelivery(@PathParam("deliveryId") String deliveryId);

    /**
     * Updates a delivery.
     *
     * @param deliveryId the delivery identifier.
     * @param delivery   new contents of the delivery.
     * @return the updated delivery.
     */
    @PUT
    @Path("/{deliveryId:.*Delivery[^/]*}")
    @PublicApiMember
    Delivery updateDelivery(@PathParam("deliveryId") String deliveryId, Delivery delivery);

    /**
     * Updates a delivery.
     *
     * @param delivery new contents of the delivery.
     * @return the updated delivery.
     */
    @PublicApiMember
    Delivery updateDelivery(Delivery delivery);

    /**
     * Deletes a delivery.
     *
     * @param deliveryId the delivery identifier.
     */
    @DELETE
    @Path("/{deliveryId:.*Delivery[^/]*}")
    @PublicApiMember
    void deleteDelivery(@PathParam("deliveryId") String deliveryId);

    /**
     * Search deliveries by filters.
     *
     * @param deliveryFilters the search criteria.
     * @param page            the page of results to return. Default value is 0.
     * @param resultsPerPage  the number of results per page. Default and maximum value is 100.
     * @param orderBy         sort options by which to sort the deliveries. Leave null for ordering by title.
     * @return the list of matching deliveries.
     */
    @POST
    @Path("search")
    @PublicApiMember
    List<Delivery> searchDeliveries(DeliveryFilters deliveryFilters,
                                    @DefaultValue("0") @QueryParam(PAGE) Long page,
                                    @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage,
                                    @DefaultValue("START_DATE") @QueryParam(ORDER_BY) DeliveryOrderMode orderBy);

    /**
     * Search deliveries by filters with default pagination.
     *
     * @param deliveryFilters the search criteria.
     * @param orderBy         sort options by which to sort the deliveries. Leave null for ordering by title.
     * @return the list of matching deliveries.
     */
    @PublicApiMember
    List<Delivery> searchDeliveries(DeliveryFilters deliveryFilters, DeliveryOrderMode orderBy);

    /**
     * Search deliveries by filters with default pagination and ordering by title.
     *
     * @param deliveryFilters the search criteria.
     * @return the list of matching deliveries.
     */
    @PublicApiMember
    List<Delivery> searchDeliveries(DeliveryFilters deliveryFilters);

    @GET
    @Path("/{deliveryId:.*Delivery[^/]*}/timeline")
    DeliveryTimeline getDeliveryTimeline(@PathParam("deliveryId") String deliveryId);

    /**
     * Returns releases associated with a given delivery.
     *
     * @param deliveryId the delivery identifier.
     * @return the release info associated with the delivery.
     */
    @GET
    @Path("/{deliveryId:.*Delivery[^/]*}/releases")
    @PublicApiMember
    List<DeliveryFlowReleaseInfo> getReleases(@PathParam("deliveryId") String deliveryId);

    // SUBSCRIBERS

    @PublicApiMember
    SubscriptionResult registerSubscriber(String deliveryId, Subscriber subscriber);

    // TRACKED ITEM ENDPOINTS

    /**
     * Adds a tracked item to a delivery.
     *
     * @param deliveryId the delivery identifier.
     * @param item       the item to be added to the delivery.
     * @return the created item.
     */
    @POST
    @Path("/{deliveryId:.*Delivery[^/]*}/tracked-items")
    @PublicApiMember
    TrackedItem createTrackedItem(@PathParam("deliveryId") String deliveryId, TrackedItem item);

    /**
     * Returns the tracked items associated with given delivery.
     *
     * @param deliveryId the delivery identifier.
     * @return tracked items associated with the delivery.
     */
    @GET
    @Path("/{deliveryId:.*Delivery[^/]*}/tracked-items")
    @PublicApiMember
    List<TrackedItem> getTrackedItems(@PathParam("deliveryId") String deliveryId);

    /**
     * Updates a tracked item.
     *
     * @param itemId the full item identifier.
     * @param item   new contents of the item.
     * @return the updated item.
     */
    @PUT
    @Path("/{itemId:.*TrackedItem[^/]*}")
    @PublicApiMember
    TrackedItem updateTrackedItem(@PathParam("itemId") String itemId, TrackedItem item);

    /**
     * Updates a tracked item.
     *
     * @param item new contents of the item.
     * @return the updated item.
     */
    @PublicApiMember
    TrackedItem updateTrackedItem(TrackedItem item);

    /**
     * Deletes a tracked item.
     *
     * @param itemId the full item identifier.
     */
    @DELETE
    @Path("/{itemId:.*TrackedItem[^/]*}")
    @PublicApiMember
    void deleteTrackedItem(@PathParam("itemId") String itemId);

    /**
     * Descopes a tracked item.
     *
     * @param itemId the full item identifier.
     */
    @PUT
    @Path("/{itemId:.*TrackedItem[^/]*}/descope")
    @PublicApiMember
    void descopeTrackedItem(@PathParam("itemId") String itemId);

    /**
     * Rescopes a tracked item.
     *
     * @param itemId the full item identifier.
     */
    @PUT
    @Path("/{itemId:.*TrackedItem[^/]*}/rescope")
    @PublicApiMember
    void rescopeTrackedItem(@PathParam("itemId") String itemId);

    /**
     * Completes a tracked item in stage.
     *
     * @param stageId the full stage identifier.
     * @param itemId  the item identifier.
     */
    @PUT
    @Path("/{stageId:.*Stage[^/]*}/{itemId:.*TrackedItem[^/]*}/complete")
    @PublicApiMember
    void completeTrackedItem(@PathParam("stageId") String stageId, @PathParam("itemId") String itemId);

    /**
     * Completes a tracked item in stage.
     *
     * @param stageId   the full stage identifier.
     * @param itemId    the item identifier.
     * @param releaseId the release identifier that completed item.
     */
    @PUT
    @Path("/{stageId:.*Stage[^/]*}/{itemId:.*TrackedItem[^/]*}/{releaseId:.*Release[^/]*}/complete")
    @PublicApiMember
    void completeTrackedItem(@PathParam("stageId") String stageId, @PathParam("itemId") String itemId, @PathParam("releaseId") String releaseId);

    /**
     * Skips a tracked item in stage.
     *
     * @param stageId the full stage identifier.
     * @param itemId  the item identifier.
     */
    @PUT
    @Path("/{stageId:.*Stage[^/]*}/{itemId:.*TrackedItem[^/]*}/skip")
    @PublicApiMember
    void skipTrackedItem(@PathParam("stageId") String stageId, @PathParam("itemId") String itemId);

    @PUT
    @Path("/{stageId:.*Stage[^/]*}/{itemId:.*TrackedItem[^/]*}/reset")
    void resetTrackedItem(@PathParam("stageId") String stageId, @PathParam("itemId") String itemId);

    @PublicApiMember
    void registerTrackedItems(String deliveryId, List<String> itemIdOrTitles, String fromReleaseId);

    @PublicApiMember
    List<TrackedItem> markTrackedItemsInStage(String deliveryId, String stageIdOrTitle, List<String> itemIdOrTitles, TrackedItemStatus status, boolean precedingStages, String releaseId);

    // STAGE ENDPOINTS

    /**
     * Returns the stages in a given delivery.
     *
     * @param deliveryId the delivery identifier.
     * @return stages in the delivery.
     */
    @GET
    @Path("/{deliveryId:.*Delivery[^/]*}/stages")
    @PublicApiMember
    List<Stage> getStages(@PathParam("deliveryId") String deliveryId);

    /**
     * Completes a stage in a delivery.
     *
     * @param stageId the full stage identifier.
     */
    @POST
    @Path("/{stageId:.*Stage[^/]*}/complete")
    @PublicApiMember
    void completeStage(@PathParam("stageId") String stageId);

    /**
     * Reopens a closed stage in a delivery.
     *
     * @param stageId the full stage identifier.
     */
    @POST
    @Path("/{stageId:.*Stage[^/]*}/reopen")
    @PublicApiMember
    void reopenStage(@PathParam("stageId") String stageId);

    /**
     * Updates a stage in a delivery.
     *
     * @param stage new contents of the stage.
     * @return the updated stage.
     */
    @PublicApiMember
    Stage updateStage(Stage stage);

    /**
     * Updates a stage in a delivery.
     *
     * @param stageId the full stage identifier.
     * @param stage   new contents of the stage.
     * @return the updated stage.
     */
    @PUT
    @Path("/{stageId:.*Stage[^/]*}")
    @PublicApiMember
    Stage updateStage(@PathParam("stageId") String stageId, Stage stage);

    // TRANSITION ENDPOINTS

    /**
     * Updates a transition on a delivery.
     *
     * @param transitionId the full transition identifier.
     * @param transition   new contents of the transition.
     * @return the updated transition
     */
    @PUT
    @Path("/{transitionId:.*Transition[^/]*}")
    @PublicApiMember
    Transition updateTransition(@PathParam("transitionId") String transitionId, Transition transition);

    /**
     * Updates a transition on a delivery.
     *
     * @param transition new contents of the transition.
     * @return the updated transition
     */
    @PublicApiMember
    Transition updateTransition(Transition transition);

    /**
     * Executes a transition on a delivery manually.
     *
     * @param transitionId       the full transition identifier.
     * @param completeTransition the transition completion parameters.
     */
    @POST
    @Path("/{transitionId:.*Transition[^/]*}/complete")
    @PublicApiMember
    void completeTransition(@PathParam("transitionId") String transitionId, CompleteTransition completeTransition);
}
