package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.UriInfo;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters;
import com.xebialabs.xlrelease.domain.Trigger;
import com.xebialabs.xlrelease.rest.AllCILevels;
import com.xebialabs.xlrelease.view.ReleaseTriggerOverview;
import com.xebialabs.xlrelease.views.BulkActionResultView;

/**
 * Operations on triggers.
 */
@Path("/api/v1/triggers")
@AllCILevels
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface TriggersApi extends ApiService {

    default String serviceName() {
        return "triggersApi";
    }

    /**
     * Searches release triggers by filters.
     *
     * @param triggerFilters query parameters derived from TriggerFilters property names used as a search criteria
     * @param page           the page of results to return. Default value is 0.
     * @param resultsPerPage the number of results per page. Default and maximum value is 100.
     * @param uriInfo        sort options encoded as URL query parameters
     *                       (https://docs.spring.io/spring-data/rest/docs/current/reference/html/#paging-and-sorting.sorting)
     * @return the list of matching release triggers
     */
    @GET
    Page<ReleaseTriggerOverview> searchTriggers(@BeanParam TriggerFilters triggerFilters,
                                                @DefaultValue("0") @QueryParam(PAGE) int page,
                                                @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage,
                                                @Context UriInfo uriInfo);

    /**
     * Gets List of Triggers with sort parameters.
     *
     * @param triggerFilters the search criteria
     * @param page           the page of results to return. Default value is 0.
     * @param resultsPerPage the number of results per page. Default and maximum value is 100.
     * @param sort           the sorting criteria
     *                       (https://docs.spring.io/spring-data/commons/docs/current/api/org/springframework/data/domain/Sort.html)
     * @return found {@code List<ReleaseTriggerOverview>} object
     */
    @PublicApiMember
    List<ReleaseTriggerOverview> searchTriggers(TriggerFilters triggerFilters, int page, int resultsPerPage, Sort sort);

    /**
     * Gets List of Triggers with default sort.
     *
     * @param triggerFilters the search criteria
     * @param page           the page of results to return. Default value is 0.
     * @param resultsPerPage the number of results per page. Default and maximum value is 100.
     * @return found {@code List<ReleaseTriggerOverview>} object
     */
    @PublicApiMember
    List<ReleaseTriggerOverview> searchTriggers(TriggerFilters triggerFilters, int page, int resultsPerPage);

    /**
     * Enables a trigger.
     *
     * @param triggerId the identifier of the trigger.
     */
    @PUT
    @Path("/{triggerId:.*/Trigger[^/]*}/enable")
    @PublicApiMember
    void enableTrigger(@PathParam("triggerId") String triggerId);

    /**
     * Enables triggers with given identifiers.
     *
     * @param triggerIds list of trigger identifiers.
     * @return list of successfully enabled trigger identifiers.
     */
    @POST
    @Path("enable")
    @PublicApiMember
    BulkActionResultView enableTriggers(List<String> triggerIds);

    @POST
    @Path("enable/all")
    @PublicApiMember
    BulkActionResultView enableAllTriggers();

    /**
     * Disables a trigger.
     *
     * @param triggerId the identifier of the trigger.
     */
    @PUT
    @Path("/{triggerId:.*/Trigger[^/]*}/disable")
    @PublicApiMember
    void disableTrigger(@PathParam("triggerId") String triggerId);

    /**
     * Disables triggers with given identifiers.
     *
     * @param triggerIds list of trigger identifiers.
     * @return list of successfully disabled trigger identifiers.
     */
    @POST
    @Path("disable")
    @PublicApiMember
    BulkActionResultView disableTriggers(List<String> triggerIds);

    @POST
    @Path("disable/all")
    @PublicApiMember
    BulkActionResultView disableAllTriggers();

    /**
     * Deletes a trigger.
     *
     * @param triggerId the identifier of the trigger.
     */
    @DELETE
    @Path("{triggerId:.*/Trigger[^/]*}")
    @PublicApiMember
    void removeTrigger(@PathParam("triggerId") String triggerId);

    /**
     * Executes trigger with given identifier.
     *
     * @param triggerId the identifier of the trigger.
     */
    @POST
    @Path("{triggerId:.*/Trigger[^/]*}/run")
    void runTrigger(@PathParam("triggerId") String triggerId);

    /**
     * Returns the list of trigger types.
     *
     * @return the list of trigger types.
     */
    @GET
    @PublicApiMember
    @Path("types")
    List<Descriptor> getTypes();

    /**
     * Create new trigger.
     *
     * @param trigger the trigger to create.
     * @return created trigger.
     */
    @POST
    @PublicApiMember
    Trigger addTrigger(Trigger trigger);

    /**
     * Returns the trigger with the given ID.
     *
     * @param triggerId the identifier of the trigger.
     * @return the trigger.
     */
    @GET
    @PublicApiMember
    @Path("{triggerId:.*Trigger[^/-]*}")
    Trigger getTrigger(@PathParam("triggerId") String triggerId);

    /**
     * Updates the properties of a trigger.
     *
     * @param triggerId the identifier of the trigger.
     * @param trigger   new contents of the trigger.
     * @return the updated trigger.
     */
    @PUT
    @PublicApiMember
    @Path("{triggerId:.*Trigger[^/-]*}")
    Trigger updateTrigger(@PathParam("triggerId") String triggerId, Trigger trigger);
}
