package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import java.util.Map;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.configuration.SystemMessageSettings;
import com.xebialabs.xlrelease.domain.Configuration;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.rest.AllCILevels;
import com.xebialabs.xlrelease.views.ConfigurationView;
import com.xebialabs.xlrelease.views.SharedConfigurationStatusResponse;

/**
 * Operations on global variables and other configuration.
 */
@Path("/api/v1/config")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
@AllCILevels
public interface ConfigurationApi extends ApiService {

    String SERVICE_NAME = "configurationApi";

    default String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Returns global variables list.
     *
     * @return variables list.
     */
    @GET
    @Path("/Configuration/variables/global")
    @PublicApiMember
    List<Variable> getGlobalVariables();

    /**
     * Returns global variables list with replaced variable values.
     *
     * @return variables list.
     */
    @GET
    @Path("/Configuration/variableValues/global")
    @PublicApiMember
    Map<String, String> getGlobalVariableValues();

    /**
     * Returns the global variable for the given identifier.
     *
     * @param variableId the variable identifier.
     * @return the variable.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}")
    @PublicApiMember
    Variable getGlobalVariable(@PathParam("variableId") String variableId);

    /**
     * Adds a global variable.
     *
     * @param variable the global variable to add.
     * @return the added global variable.
     */
    @POST
    @Path("/Configuration/variables/global")
    @PublicApiMember
    Variable addGlobalVariable(com.xebialabs.xlrelease.api.v1.forms.Variable variable);

    @PUT
    @Path("/{variableId:.*/Variable[^/]*}")
    Variable updateGlobalVariable(@PathParam("variableId") String variableId, Variable variable);

    /**
     * Updates properties of a global variable by its ID.
     *
     * @param variable the variable to update.
     * @return the updated variable.
     */
    @PublicApiMember
    Variable updateGlobalVariable(Variable variable);

    /**
     * Deletes a global variable.
     *
     * @param variableId the variable identifier.
     */
    @DELETE
    @Path("/{variableId:.*/Variable[^/]*}")
    @PublicApiMember
    void deleteGlobalVariable(@PathParam("variableId") String variableId);

    /**
     * Searches configuration objects by type and title.
     *
     * @param configurationType the type of the Configuration object.
     * @param title             the title of the Configuration object.(optional)
     * @return returns the first 100 results loaded with depth 1. Password properties are not returned.
     */
    @PublicApiMember
    List<? extends ConfigurationItem> searchByTypeAndTitle(final String configurationType, final String title);

    /**
     * Searches configuration objects by type and title.
     *
     * @param configurationType the type of the Configuration object.
     * @param title             the title of the Configuration object.(optional)
     * @param folderId          the path of a parent.(optional)
     * @return returns the first 100 results loaded with depth 1. Password properties are not returned.
     */
    @PublicApiMember
    List<? extends ConfigurationItem> searchByTypeAndTitle(final String configurationType, final String title, final String folderId);

    /**
     * Searches configuration objects by type and title.
     *
     * @param configurationType the type of the Configuration object.
     * @param title             the title of the Configuration object.(optional)
     * @param folderId          the path of a parent.(optional)
     * @param folderOnly        filter if you only want to get information from your parentId. (optional)
     * @return returns the first 100 results loaded with depth 1. Password properties are not returned.
     */
    @GET
    @Path("byTypeAndTitle")
    @PublicApiMember
    List<Configuration> searchByTypeAndTitle(
            @QueryParam("configurationType") String configurationType,
            @QueryParam("title") String title,
            @QueryParam("folderId") String folderId,
            @QueryParam("folderOnly") boolean folderOnly);


    /**
     * Returns the list of configurations for the given list of IDs.
     *
     * @param configurationIds the list of IDs for configuration items to fetch
     * @return the list of configuration instances
     */
    @POST
    @Path("byIds")
    @PublicApiMember
    List<Configuration> getConfigurations(List<String> configurationIds);

    /**
     * Returns the configuration with the given ID.
     *
     * @param configurationId the identifier of the configuration
     * @return the configuration instance
     */
    @GET
    @Path("/{configurationId:.*/Configuration[^/]*}")
    @PublicApiMember
    Configuration getConfiguration(@PathParam("configurationId") String configurationId);

    /**
     * Creates a new configuration.
     *
     * @param configuration the configuration to add.
     * @return the created configuration.
     */
    @POST
    @PublicApiMember
    Configuration addConfiguration(Configuration configuration);

    /**
     * Updates the configuration.
     *
     * @param configurationId the identifier of the configuration.
     * @param configuration   new contents of the configuration.
     * @return the updated configuration.
     */
    @PUT
    @Path("{configurationId:.*/Configuration[^/]*}")
    Configuration updateConfiguration(@PathParam("configurationId") String configurationId, Configuration configuration);

    /**
     * Updates the configuration.
     *
     * @param configuration new contents of the configuration.
     * @return the updated configuration.
     */
    @PublicApiMember
    Configuration updateConfiguration(Configuration configuration);

    /**
     * Deletes the configuration.
     *
     * @param configurationId the identifier of the configuration.
     */
    @DELETE
    @Path("{configurationId:.*/Configuration[^/]*}")
    @PublicApiMember
    void deleteConfiguration(@PathParam("configurationId") String configurationId);

    @GET
    @Path("system-message")
    @PublicApiMember
    SystemMessageSettings getSystemMessage();

    @PUT
    @Path("system-message")
    @PublicApiMember
    SystemMessageSettings updateSystemMessage(final SystemMessageSettings systemMessageSettings);

    @POST
    @Path("status")
    @PublicApiMember
    SharedConfigurationStatusResponse checkStatus(final ConfigurationView configurationView);

    @POST
    @Path("{configurationId:.*/Configuration[^/]*}/status")
    @PublicApiMember
    SharedConfigurationStatusResponse checkStatus(final @PathParam("configurationId") String configurationId);

}
