package com.xebialabs.xlrelease.api.v1;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.AbortRelease;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.api.v1.views.BasicReleaseView;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.PhaseVersion;
import com.xebialabs.xlrelease.rest.AllCILevels;
import com.xebialabs.xlrelease.search.ReleaseCountResults;
import com.xebialabs.xlrelease.search.ReleaseFullSearchResult;

/**
 * Operations on releases.
 */
@Path("/api/v1/releases")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@AllCILevels
@PublicApi
@ShowOnlyPublicApiMembers
public interface ReleaseApi extends ApiService {

    String SERVICE_NAME = "releaseApi";
    long DEFAULT_PAGE = 0L;
    String ARCHIVE_PAGE = "archivePage";
    String ARCHIVE_RESULTS_PER_PAGE = "archiveResultsPerPage";

    default String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Returns the attachment file with the given ID.
     *
     * @param attachmentId the identifier of the attachment
     * @return the attachment file.
     */
    @GET
    @Path("attachments/{attachmentId:.*/Attachment[^/]*}")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    Response downloadAttachment(@PathParam("attachmentId") String attachmentId);

    /**
     * Returns the attachment file with the given ID.
     *
     * @param attachmentId the identifier of the attachment
     * @return the attachment file.
     */
    @PublicApiMember
    byte[] getAttachment(String attachmentId) throws IOException;


    /**
     * Count releases matching filter criteria.
     *
     * @param releasesFilters the search criteria
     * @return a map containing the number of releases (total) and the number by status
     */
    @POST
    @Path("count")
    ReleaseCountResults countReleases(ReleasesFilters releasesFilters);


    /**
     * Searches releases by filters.
     *
     * @param releasesFilters the search criteria
     * @param page            the page of results to return. Default value is 0.
     * @param resultsPerPage  the number of results per page. Default and maximum value is 100.
     * @param pageIsOffset    the flag indicating if page is used as offset. Default is false.
     * @return the list of matching releases
     */
    @POST
    @Path("search")
    @PublicApiMember
    List<Release> searchReleases(ReleasesFilters releasesFilters,
                                 @DefaultValue("0") @QueryParam(PAGE) Long page,
                                 @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage,
                                 @DefaultValue("false") @QueryParam(PAGE_IS_OFFSET) Boolean pageIsOffset);

    /**
     * Searches releases by filters.
     *
     * @param releasesFilters the search criteria
     * @param page            the page of results to return.
     * @param resultsPerPage  the number of results per page.
     * @return the list of matching releases
     */
    @PublicApiMember
    List<Release> searchReleases(ReleasesFilters releasesFilters, Long page, Long resultsPerPage);

    /**
     * Searches releases by filters with default pagination.
     *
     * @param releasesFilters the search criteria
     * @return the list of first 100 matching releases
     */
    @PublicApiMember
    List<Release> searchReleases(ReleasesFilters releasesFilters);

    /**
     * Searches releases overview by filters.
     *
     * @param releasesFilters the search criteria
     * @param page            the page of results to return. Default value is 0.
     * @param resultsPerPage  the number of results per page. Default and maximum value is 100.
     * @return the list of matching releases
     */
    @POST
    @Path("search/overview")
    @PublicApiMember
    List<BasicReleaseView> searchReleasesOverview(ReleasesFilters releasesFilters,
                                                  @DefaultValue("0") @QueryParam(PAGE) Long page,
                                                  @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage);

    /**
     * Searches releases.
     *
     * @param page                  next page to query, active database
     * @param archivePage           next page to query, archive database
     * @param resultsPerPage        releases per page, active database
     * @param archiveResultsPerPage releases per page, archive database
     * @param releasesFilters       the search criteria
     * @return the list of matching releases
     */
    @POST
    @Path("fullSearch")
    ReleaseFullSearchResult fullSearchReleases(@QueryParam(PAGE) Long page,
                                               @QueryParam(ARCHIVE_PAGE) Long archivePage,
                                               @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage,
                                               @QueryParam(ARCHIVE_RESULTS_PER_PAGE) Long archiveResultsPerPage,
                                               ReleasesFilters releasesFilters);

    /**
     * Returns the list of planned or active releases that are visible to the current user.
     * <p>
     * Active releases are ordered by the start date (ascending) and title (ascending).
     *
     * @param page           the page of results to return.
     * @param resultsPerPage the number of results per page.
     * @param depth          the depth to search for.
     * @return a list of releases.
     * @deprecated Since 24.1.0. Please use {@link #getReleases(Long, Long)}
     */
    @PublicApiMember
    List<Release> getReleases(Long page, Long resultsPerPage, Integer depth);

    /**
     * Returns the list of planned or active releases that are visible to the current user.
     * <p>
     * Active releases are ordered by the start date (ascending) and title (ascending).
     *
     * @param page           the page of results to return. Default value is 0.
     * @param resultsPerPage the number of results per page. Default and maximum value is 100.
     * @return a list of releases.
     */
    @GET
    @Path("/")
    @PublicApiMember
    List<Release> getReleases(@DefaultValue("0") @QueryParam(PAGE) Long page,
                              @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage);

    /**
     * Returns the list of the first 100 planned or active releases that are visible to the current user.
     *
     * @deprecated Since 24.1.0. Please use {@link #getReleases(Long, Long)}
     */
    @Deprecated(since = "24.1.0")
    @PublicApiMember
    List<Release> getReleases();

    /**
     * Returns the release with the given ID.
     *
     * @param releaseId   the identifier of the release.
     * @param withRoleIds if true exposes roleIds in response.
     * @return the release.
     */
    @GET
    @Path("/{releaseId:((?!archived).)*Release[^/]*}")
    @PublicApiMember
    Release getRelease(@PathParam("releaseId") String releaseId, @DefaultValue("false") @QueryParam(ROLE_IDS_DATA) boolean withRoleIds);

    /**
     * Returns the release with the given ID.
     *
     * @param releaseId the identifier of the release.
     * @return the release.
     */
    @PublicApiMember
    Release getRelease(String releaseId);

    /**
     * Returns the archived release for the given ID.
     *
     * @param releaseId   the identifier of the release.
     * @param withRoleIds if true exposes roleIds in response.
     * @return the release.
     */
    @GET
    @Path("/archived/{releaseId:.*Release[^/]*}")
    @PublicApiMember
    Release getArchivedRelease(@PathParam("releaseId") String releaseId, @DefaultValue("false") @QueryParam(ROLE_IDS_DATA) boolean withRoleIds);

    /**
     * Returns the archived release for the given ID.
     *
     * @param releaseId the identifier of the release.
     * @return the release.
     */
    @PublicApiMember
    Release getArchivedRelease(String releaseId);

    /**
     * Returns the active tasks in a release.
     *
     * @param releaseId the identifier of the release
     * @return a list of active tasks.
     */
    @GET
    @Path("/{releaseId:.*Release[^/]*}/active-tasks")
    @PublicApiMember
    List<Task> getActiveTasks(@PathParam("releaseId") String releaseId);

    /**
     * Starts a planned release.
     *
     * @param releaseId the release identifier.
     * @return the started release.
     */
    @POST
    @Path("/{releaseId:.*Release[^/]*}/start")
    @PublicApiMember
    Release start(@PathParam("releaseId") String releaseId);

    /**
     * Updates the properties of a release.
     *
     * @param releaseId the identifier of the release.
     * @param release   new contents of the release.
     * @return the updated release.
     */
    @PUT
    @Path("/{releaseId:.*Release[^/]*}")
    Release updateRelease(@PathParam("releaseId") String releaseId, Release release);

    /**
     * Updates the properties of a release.
     *
     * @param release new contents of the release.
     * @return the updated release.
     */
    @PublicApiMember
    Release updateRelease(Release release);

    /**
     * Deletes a release. The release MUST be in a final state (complete or aborted).
     *
     * @param releaseId the identifier of the release
     */
    @DELETE
    @Path("/{releaseId:.*Release[^/]*}")
    @PublicApiMember
    void delete(@PathParam("releaseId") String releaseId);

    /**
     * Aborts a release.
     *
     * @param releaseId    the identifier of the release
     * @param abortRelease the parameters to abort the release
     * @return the aborted release
     */
    @POST
    @Path("/{releaseId:.*Release[^/]*}/abort")
    Release abort(@PathParam("releaseId") String releaseId, AbortRelease abortRelease);

    /**
     * Aborts a release.
     *
     * @param releaseId    the identifier of the release
     * @param abortComment the comment of the abort
     * @return the aborted release
     */
    @PublicApiMember
    Release abort(String releaseId, String abortComment);

    /**
     * Search releases by given title (exact match).
     *
     * @param releaseTitle The title of the release.
     * @return a list of releases.
     */
    @GET
    @Path("/byTitle")
    @PublicApiMember
    List<Release> searchReleasesByTitle(@QueryParam("releaseTitle") String releaseTitle);

    /**
     * Get release variables.
     *
     * @param releaseId the identifier of the release.
     * @return a list of variables.
     */
    @GET
    @Path("/{releaseId:.*Release[^/]*}/variables")
    @PublicApiMember
    List<Variable> getVariables(@PathParam("releaseId") String releaseId);

    /**
     * Get release variable values.
     *
     * @param releaseId the identifier of the release.
     * @return a map of variable values.
     */
    @GET
    @Path("/{releaseId:.*Release[^/]*}/variableValues")
    @PublicApiMember
    Map<String, String> getVariableValues(@PathParam("releaseId") String releaseId);

    /**
     * Returns the variable for the given identifier.
     *
     * @param variableId the variable identifier.
     * @return the variable.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}")
    @PublicApiMember
    Variable getVariable(@PathParam("variableId") String variableId);

    /**
     * Returns possible values for the variable with the given identifier.
     *
     * @param variableId the variable identifier.
     * @return possible values for the variable.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}/possibleValues")
    @PublicApiMember
    Collection<Object> getVariablePossibleValues(@PathParam("variableId") String variableId);

    /**
     * Returns true if variable with the given identifier is used in the release.
     *
     * @param variableId the variable ID.
     * @return true if variable with the given identifier is used in the release.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}/used")
    @PublicApiMember
    Boolean isVariableUsed(@PathParam("variableId") String variableId);

    /**
     * Replace variable occurrences with the given replacement.
     *
     * @param variableId      the variable ID.
     * @param variableOrValue an object with a variable or value replacing the initial variable.
     */
    @POST
    @Path("/{variableId:.*/Variable[^/]*}/replace")
    @PublicApiMember
    void replaceVariable(@PathParam("variableId") String variableId, VariableOrValue variableOrValue);

    /**
     * Delete variable from release.
     *
     * @param variableId the variable ID.
     */
    @DELETE
    @Path("/{variableId:.*/Variable[^/]*}")
    @PublicApiMember
    void deleteVariable(@PathParam("variableId") String variableId);

    /**
     * Create new variable.
     *
     * @param releaseId the identifier of the release.
     * @param variable  the variable to create.
     * @return created variable.
     */
    @POST
    @Path("/{releaseId:.*?}/variables")
    @PublicApiMember
    Variable createVariable(@PathParam("releaseId") String releaseId, com.xebialabs.xlrelease.api.v1.forms.Variable variable);

    /**
     * Update the list of variables in a release.
     *
     * @param releaseId the identifier of the release.
     * @param variables the variable list to update.
     * @return updated variables.
     */
    @PUT
    @Path("/{releaseId:.*?}/variables")
    @PublicApiMember
    List<Variable> updateVariables(@PathParam("releaseId") String releaseId, List<Variable> variables);

    @PUT
    @Path("/{variableId:.*/Variable[^/]*}")
    Variable updateVariable(@PathParam("variableId") String variableId, Variable variable);

    /**
     * Updates the properties of a variable.
     *
     * @param variable new contents of the variable.
     * @return the updated variable.
     */
    @PublicApiMember
    Variable updateVariable(Variable variable);

    /**
     * Returns possible permissions.
     *
     * @return list of permissions for releases
     */
    @GET
    @Path("/permissions")
    @PublicApiMember
    List<String> getPermissions();

    /**
     * Returns effective teams of the release.
     *
     * @param releaseId the identifier of the release
     * @return a list of effective teams
     */
    @GET
    @Path("/{releaseId:.*Release[^/]*}/teams")
    @PublicApiMember
    List<TeamView> getTeams(@PathParam("releaseId") String releaseId);

    /**
     * Sets teams of the release.
     *
     * @param releaseId the identifier of the release
     * @return a list of updated teams
     */
    @POST
    @Path("/{releaseId:.*Release[^/]*}/teams")
    @PublicApiMember
    List<TeamView> setTeams(@PathParam("releaseId") String releaseId, List<TeamView> teams);

    /**
     * Resumes a release that has been paused as part of a restart phases operation.
     *
     * @param releaseId the identifier of the release
     * @return the release
     */
    @POST
    @Path("/{releaseId:.*Release[^/]*}/resume")
    @PublicApiMember
    Release resume(@PathParam("releaseId") String releaseId);

    /**
     * Restarts the release from a given phase and task.
     *
     * @param releaseId     the identifier of the release
     * @param phaseId       the identifier of the phase to restart from
     * @param taskId        the identifier of the task to restart from
     * @param phaseVersion  provide which 'version' of the phase should be copied
     * @param resumeRelease if true resumes the paused release immediately
     * @return the release
     */
    @POST
    @Path("/{releaseId:.*Release[^/]*}/restart")
    Release restartPhases(@PathParam("releaseId") String releaseId,
                          @QueryParam("fromPhaseId") String phaseId,
                          @QueryParam("fromTaskId") String taskId,
                          @QueryParam("phaseVersion") PhaseVersion phaseVersion,
                          @QueryParam("resume") boolean resumeRelease);

    /**
     * Restart release from the current phase.
     *
     * @param release the release.
     * @return the release.
     */
    @PublicApiMember
    Release restartPhase(Release release);

    /**
     * Restart release from the current phase and optionally resume the release.
     *
     * @param release       the release.
     * @param resumeRelease if true resumes the paused release immediately
     * @return the release.
     */
    @PublicApiMember
    Release restartPhase(Release release, boolean resumeRelease);

    /**
     * Restart the release from the given phase. Creates copies of all phases between the given phase and the current phase.
     * <p>
     * Will copy all phases between given Phase and the active phase of the release. If you want to only copy the latest phases, use method below
     * to influence this behavior.
     *
     * @param release the release
     * @param phase   the phase to restart from
     * @return the release
     */
    @PublicApiMember
    Release restartPhase(Release release, Phase phase);

    /**
     * Restart the release from the given phase. Depending on 'phaseVersion' it will have a different copy strategy.
     *
     * @param release      the release
     * @param phase        the phase to restart from
     * @param phaseVersion the given phase version
     * @return the release
     */
    @PublicApiMember
    Release restartPhase(Release release, Phase phase, PhaseVersion phaseVersion);

    /**
     * Restart the release from the given phase and task.
     * <p>
     * Will copy all phases between given Phase and the active phase of the release. If you want to only copy the latest phases, use method below
     * to influence this behavior.
     *
     * @param release the release
     * @param phase   the phase to restart from
     * @param task    the task to restart from
     * @return the release
     */
    @PublicApiMember
    Release restartPhase(Release release, Phase phase, Task task);

    /**
     * Restart the release from the given phase and task.
     *
     * @param release      the release
     * @param phase        the phase to restart from
     * @param task         the task to restart from
     * @param phaseVersion the given phase version
     * @return the release
     */
    @PublicApiMember
    Release restartPhase(Release release, Phase phase, Task task, PhaseVersion phaseVersion);

    /**
     * Restart the release from the given phase and task and optionally resume the release.
     *
     * @param release       the release
     * @param phase         the phase to restart from
     * @param task          the task to restart from
     * @param phaseVersion  the given phase version
     * @param resumeRelease if true resumes the paused release immediately
     * @return the release
     */
    @PublicApiMember
    Release restartPhase(Release release, Phase phase, Task task, PhaseVersion phaseVersion, boolean resumeRelease);
}
