package com.xebialabs.xlrelease.api.v1;

import java.io.IOException;
import java.util.List;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.Comment;
import com.xebialabs.xlrelease.api.v1.forms.Condition;
import com.xebialabs.xlrelease.api.v1.forms.StartTask;
import com.xebialabs.xlrelease.domain.Attachment;
import com.xebialabs.xlrelease.domain.Dependency;
import com.xebialabs.xlrelease.domain.GateCondition;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.rest.AllCILevels;

/**
 * Operations on tasks.
 */
@Path("/api/v1/tasks/")
@AllCILevels
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface TaskApi extends ApiService {

    String SERVICE_NAME = "taskApi";

    default String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Adds one or more attachment files to a task, serialized in XLR format.
     *
     * @param taskId  the task identifier.
     * @param request multipart/form-data request with attachment file.
     * @return the domain objects of the uploaded attachments.
     */
    @POST
    @Path("{taskId:.*/Task[^/]*}/attachments")
    @Consumes({MediaType.MULTIPART_FORM_DATA})
    List<Attachment> addAttachments(@PathParam("taskId") String taskId,
                                    @Context HttpServletRequest request) throws IOException;

    /**
     * Adds one attachment files to a task.
     */
    @PublicApiMember
    Attachment addAttachment(String taskId, String fileName, byte[] fileByteArray) throws IOException;

    /**
     * Deletes an attachment from a task.
     *
     * @param taskId       the task identifier.
     * @param attachmentId the attachment identifier.
     */
    @DELETE
    @Path("{taskId:.*/Task[^/]*}/attachments/{attachmentId:.*/Attachment[^/]*}")
    @PublicApiMember
    void deleteAttachment(@PathParam("taskId") String taskId, @PathParam("attachmentId") String attachmentId);

    /**
     * Returns a task by ID.
     *
     * @param taskId the task identifier
     * @return the task which has the given identifier.
     */
    @GET
    @Path("/{taskId:.*/Task[^/]*}")
    @PublicApiMember
    Task getTask(@PathParam("taskId") String taskId);

    /**
     * Copy a task
     *
     * @param taskId            the task identifier.
     * @param targetContainerId identifier of the target task container.
     * @param targetPosition    the position where to copy the task to. Cannot be {@code < 0} and also {@code not >= container.size()}
     * @return the copy of the task
     */
    @POST
    @Path("{taskId:.*/Task[^/]*}/copy")
    @PublicApiMember
    Task copyTask(@PathParam("taskId") String taskId,
                  @QueryParam("targetContainerId") String targetContainerId,
                  @QueryParam("targetPosition") int targetPosition);

    /**
     * Updates a task.
     *
     * @param task new contents of the task.
     * @return the updated task.
     */
    @PublicApiMember
    Task updateTask(Task task);

    /**
     * Updates a task.
     *
     * @param taskId the task identifier.
     * @param task   new contents of the task.
     * @return the updated task.
     */
    @PUT
    @Path("/{taskId:.*/Task[^/]*}")
    Task updateTask(@PathParam("taskId") String taskId, Task task);

    /**
     * Updates a task.
     *
     * @param taskId       the task identifier.
     * @param task         new contents of the task.
     * @param overrideLock enables override of locked task.
     * @return the updated task.
     */
    @PublicApiMember
    Task updateTask(String taskId, Task task, boolean overrideLock);

    /**
     * Changes a task type to the target type. Target type can be one of core task types
     * such as "xlrelease.GateTask" or a custom script definition type, such as "jenkins.Build".
     *
     * @param taskId     the identifier of the task.
     * @param targetType the new task type.
     * @return task with new type.
     */
    @POST
    @Path("{taskId:.*/Task[^/]*}/changeType")
    @PublicApiMember
    Task changeTaskType(@PathParam("taskId") String taskId, @QueryParam("targetType") String targetType);

    /**
     * Adds a task to a phase or a container task.
     *
     * @param containerId the identifier of the task container: either a {@link com.xebialabs.xlrelease.domain.Phase},
     *                    a {@link com.xebialabs.xlrelease.domain.ParallelGroup} or a {@link com.xebialabs.xlrelease.domain.SequentialGroup}.
     * @param task        the task to add.
     * @return the added task.
     */
    @POST
    @Path("/{containerId:.*?}/tasks")
    @PublicApiMember
    Task addTask(@PathParam("containerId") String containerId, Task task);

    /**
     * Adds a task to a phase or a container task.
     *
     * @param containerId the identifier of the task container: either a {@link com.xebialabs.xlrelease.domain.Phase},
     *                    a {@link com.xebialabs.xlrelease.domain.ParallelGroup} or a {@link com.xebialabs.xlrelease.domain.SequentialGroup}.
     * @param task        the task to add.
     * @param position    the position at which the task will be added.
     * @return the added task.
     */
    @POST
    @Path("/{containerId:.*?}/tasks")
    Task addTask(@PathParam("containerId") String containerId, Task task, Integer position);

    /**
     * Completes a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'complete' action.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/complete")
    @PublicApiMember
    Task completeTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Completes a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'complete' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task completeTask(String taskId, String comment);

    /**
     * Skips a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'skip' action.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/skip")
    @PublicApiMember
    Task skipTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Skips a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'skip' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task skipTask(String taskId, String comment);

    /**
     * Fails a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'fail' action.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/fail")
    @PublicApiMember
    Task failTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Fails a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'fail' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task failTask(String taskId, String comment);

    /**
     * Aborts a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'abort' action.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/abort")
    @PublicApiMember
    Task abortTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Aborts a task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'abort' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task abortTask(String taskId, String comment);

    /**
     * Retry failed task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'retry' action.
     * @return the updated task.
     */
    @PublicApiMember
    @POST
    @Path("/{taskId:.*/Task[^/]*}/retry")
    Task retryTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Retry failed task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'retry' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task retryTask(String taskId, String comment);

    /**
     * Start pending task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'startNow' action.
     * @return the updated task.
     */
    @PublicApiMember
    @POST
    @Path("/{taskId:.*/Task[^/]*}/startNow")
    Task start(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Start pending task.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'startNow' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task start(String taskId, String comment);

    /**
     * Starts a task that is waiting for input.
     *
     * @param task      the task to be started.
     * @param variables the input variables required to start a task.
     * @return the updated task.
     */
    @PublicApiMember
    Task start(Task task, List<Variable> variables);

    /**
     * Starts a task that is waiting for input.
     *
     * @param taskId    the task identifier.
     * @param startTask the object with parameters required to start a task, e.g. variables.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/start")
    Task start(@PathParam("taskId") String taskId, StartTask startTask);

    /**
     * Reopen a task that has been completed in advance.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'reopen' action.
     * @return the updated task.
     */
    @PublicApiMember
    @POST
    @Path("/{taskId:.*/Task[^/]*}/reopen")
    Task reopenTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Reopen a task that has been completed in advance.
     *
     * @param taskId  the task identifier.
     * @param comment a comment describing the 'reopen' action.
     * @return the updated task.
     */
    @PublicApiMember
    Task reopenTask(String taskId, String comment);

    /**
     * Return the variables used by the task input fields.
     *
     * @param taskId the task identifier.
     * @return input variables used by the task.
     */
    @GET
    @Path("/{taskId:.*/Task[^/]*}/variables")
    @PublicApiMember
    List<Variable> getVariables(@PathParam("taskId") String taskId);

    /**
     * Updates the variables values of a UserInputTask
     * It doesn't remove or add new variables
     *
     * @param taskId the task identifier.
     * @param variables the variables with values to be updated.
     * @return the updated variables.
     */
    @PUT
    @Path("/{taskId:.*/Task[^/]*}/variables")
    @PublicApiMember
    List<Variable> updateInputVariables(@PathParam("taskId") String taskId, List<Variable> variables);


    /**
     * Adds a comment to a task.
     *
     * @param taskId  the task identifier.
     * @param comment the comment text to be added to the task.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/comment")
    @PublicApiMember
    Task commentTask(@PathParam("taskId") String taskId, Comment comment);

    /**
     * Adds a comment to a task.
     *
     * @param taskId  the task identifier.
     * @param comment the comment text to be added to the task.
     * @return the updated task.
     */
    @PublicApiMember
    Task commentTask(String taskId, String comment);

    /**
     * Assigns a task to a user.
     *
     * @param taskId   the task identifier.
     * @param username the username of the user the task will be assigned to.
     * @return the updated task.
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/assign/{username: .*?}")
    @PublicApiMember
    Task assignTask(@PathParam("taskId") String taskId, @PathParam("username") String username);

    /**
     * Search tasks by title within a release.
     *
     * @param taskTitle  the task title.
     * @param phaseTitle the phase title (optional).
     * @param releaseId  the release identifier.
     * @return a List of tasks that match the title within the release.
     */
    @GET
    @Path("/byTitle")
    @PublicApiMember
    List<Task> searchTasksByTitle(@QueryParam("taskTitle") String taskTitle,
                                  @QueryParam("phaseTitle") String phaseTitle,
                                  @QueryParam("releaseId") String releaseId);


    /**
     * Factory method to create a new instance of a Manual Task.
     *
     * @return the task object.
     */
    @PublicApiMember
    Task newTask();

    /**
     * Factory method to create a new instance of a Task object.
     * <p>
     * Example script which adds a custom script task to the current phase:
     * <pre>
     * {@code
     * phase = getCurrentPhase()
     * task = taskApi.newTask("xlrelease.GateTask")
     * task.title = "My Gate Task"
     * taskApi.addTask(phase.id, task)
     * }
     * </pre>
     *
     * @param type the type of the task. For example: {@code xlrelease.GateTask}.
     * @return the task object.
     */
    @PublicApiMember
    Task newTask(String type);

    /**
     * Factory method to create a new instance of a Comment.
     *
     * @param commentText comment text
     * @return the comment object.
     */
    Comment newComment(String commentText);

    /**
     * Deletes a task.
     *
     * @param taskId the task identifier.
     */
    @DELETE
    @Path("/{taskId:.*/Task[^/]*}")
    @PublicApiMember
    void delete(@PathParam("taskId") String taskId);

    /**
     * Deletes a dependency from a gate task.
     *
     * @param dependencyId the identifier of the gate task dependency.
     */
    @DELETE
    @Path("/{dependencyId:.*/Dependency[^/]*}")
    @PublicApiMember
    void deleteDependency(@PathParam("dependencyId") String dependencyId);

    /**
     * Adds a dependency to a gate task.
     *
     * @param taskId   the identifier of the gate task to add the dependency to.
     * @param targetId the identifier of the release, phase or task that the gate will depend on, or a variable which will resolve to such an identifier.
     *                 When using a variable you can also combine variables, for example "Applications/${shortReleaseId}"
     * @return the added Dependency
     */
    @POST
    @Path("/{taskId:.*/Task[^/]*}/dependencies/{targetId:.*?}")
    @PublicApiMember
    Dependency addDependency(@PathParam("taskId") String taskId, @PathParam("targetId") String targetId);

    /**
     * Adds a condition to a gate task.
     *
     * @param taskId    the identifier of the gate task to add the condition to.
     * @param condition the condition to add.
     * @return the added gate condition.
     */
    @POST
    @Path("{taskId:.*/Task[^/]*}/conditions")
    @PublicApiMember
    GateCondition addCondition(@PathParam("taskId") String taskId, Condition condition);

    /**
     * Updates a condition on a gate task. Use this method to (un)check the condition.
     *
     * @param conditionId the identifier of the gate task condition.
     * @param condition   the new values of the condition.
     * @return the updated gate condition.
     */
    @POST
    @Path("/{conditionId:.*/GateCondition[^/]*}")
    GateCondition updateCondition(@PathParam("conditionId") String conditionId, Condition condition);

    /**
     * Updates a condition on a gate task. Use this method to (un)check the condition.
     *
     * @param condition the new values of the condition.
     * @return the updated gate condition.
     */
    @PublicApiMember
    GateCondition updateCondition(GateCondition condition);

    /**
     * Deletes a condition from a gate task.
     *
     * @param conditionId the identifier of the gate task condition.
     */
    @DELETE
    @Path("{conditionId:.*/GateCondition[^/]*?}")
    @PublicApiMember
    void deleteCondition(@PathParam("conditionId") String conditionId);

    /**
     * Locks a task.
     *
     * @param taskId the task identifier.
     */
    @PUT
    @PublicApiMember
    @Path("{taskId:.*/Task[^/]*}/lock")
    void lockTask(@PathParam("taskId") final String taskId);

    /**
     * Unlocks a task.
     *
     * @param taskId the task identifier.
     */
    @DELETE
    @PublicApiMember
    @Path("{taskId:.*/Task[^/]*}/lock")
    void unlockTask(@PathParam("taskId") final String taskId);
}
