package com.xebialabs.xlrelease.api.v1;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.UserAccount;
import com.xebialabs.xlrelease.param.DateParam;
import com.xebialabs.xlrelease.rest.AllCILevels;
import com.xebialabs.xlrelease.views.ChangePasswordView;

/**
 * Operations with users.
 */
@Path("/api/v1/users")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@AllCILevels
@PublicApi
@ShowOnlyPublicApiMembers
public interface UserApi extends ApiService {

    default String serviceName() {
        return "userApi";
    }

    // REST API

    @POST
    @Path("/{username:[^/]+}")
    Response createUser(@PathParam("username") String username, UserAccount userAccount);

    @PUT
    @Path("/{username:.*}")
    Response updateUser(@PathParam("username") String username, UserAccount userAccount);

    @PUT
    Response updateUsersRest(Collection<UserAccount> userAccounts);

    @POST
    @Path("/{username:[^/]+}/password")
    Response updatePassword(@PathParam("username") String username, ChangePasswordView changePasswordView);

    @DELETE
    @Path("/{username:.*}")
    Response deleteUserRest(@PathParam("username") String username);


    // REST+Jython API

    /**
     * Returns the users in the system.
     *
     * @param email          The (optional) email of the user
     * @param fullName       The (optional) full name of the user
     * @param loginAllowed   The (optional) flag for login permission
     * @param external       The (optional) flag for external users
     * @param page           The (optional) page of results to return. Default value is 0
     * @param resultsPerPage The (optional) number of results per page. Default value is 100
     * @return users found
     */
    @GET
    @PublicApiMember
    List<UserAccount> findUsers(@QueryParam("email") String email,
                                @QueryParam("fullName") String fullName,
                                @QueryParam("loginAllowed") Boolean loginAllowed,
                                @QueryParam("external") Boolean external,
                                @QueryParam("lastActiveAfter") @DateParam Date lastActiveAfter,
                                @QueryParam("lastActiveBefore") @DateParam Date lastActiveBefore,
                                @DefaultValue("0") @QueryParam(PAGE) Long page,
                                @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage);

    /**
     * Returns a single user
     *
     * @param username the identifier of the user
     * @return user found
     */
    @GET
    @Path("/{username:.*}")
    @PublicApiMember
    UserAccount getUser(@PathParam("username") String username);


    // Jython API

    /**
     * Creates a new user
     *
     * @param userAccount The user properties
     * @return the new user created
     */
    @PublicApiMember
    UserAccount createUser(UserAccount userAccount);

    /**
     * Updates a user.
     *
     * @param userAccount The user properties
     * @return the updated user
     */
    @PublicApiMember
    UserAccount updateUser(UserAccount userAccount);

    /**
     * Updates a collection of users
     *
     * @param userAccounts Collection of user properties
     */
    @PublicApiMember
    void updateUsers(Collection<UserAccount> userAccounts);

    /**
     * Updates a user password. External users can't be updated with this operation.
     *
     * @param username        The identifier of the user.
     * @param currentPassword Current password of the user, it only has to be supplied when changing your own password.
     * @param newPassword     The new password.
     */
    @PublicApiMember
    void updatePassword(String username, String currentPassword, String newPassword);

    /**
     * Deletes a user.
     *
     * @param username The identifier of the user
     */
    @PublicApiMember
    void deleteUser(String username);
}
