package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.repository.PhaseVersion;
import com.xebialabs.xlrelease.rest.AllCILevels;

/**
 * Operations on phases.
 */
@Path("/api/v1/phases")
@AllCILevels
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface PhaseApi extends ApiService {

    String SERVICE_NAME = "phaseApi";

    default String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Returns the phase for the given identifier.
     *
     * @param phaseId the phase identifier.
     * @return the phase.
     */
    @GET
    @Path("/{phaseId:.*/Phase[^/]*}")
    @PublicApiMember
    Phase getPhase(@PathParam("phaseId") String phaseId);

    /**
     * Updates the properties of a phase.
     *
     * @param phaseId the phase identifier.
     * @param phase   new contents of the phase.
     * @return the updated phase
     */
    @PUT
    @Path("/{phaseId:.*/Phase[^/]*}")
    Phase updatePhase(@PathParam("phaseId") String phaseId, Phase phase);

    /**
     * Updates the properties of a phase.
     *
     * @param phase new contents of the phase.
     * @return the updated Phase
     */
    @PublicApiMember
    Phase updatePhase(Phase phase);

    /**
     * Adds a task to a phase or to a group task. If no position is specified, then task is added to the end
     * of the phase.
     *
     * @param containerId the phase or group task identifier
     * @param task        the task to add.
     * @param position    the position in the phase.task, 0-based. If left empty, then task will
     *                    be added to the end of the phase.
     * @return the added task
     */
    @POST
    @Path("/{containerId:.*/Phase.*?}/tasks")
    @PublicApiMember
    Task addTask(@PathParam("containerId") String containerId, Task task, @QueryParam("position") Integer position);

    /**
     * Search phases in a release by title.
     *
     * @param phaseTitle the phase title
     * @param releaseId  the release identifier.
     * @return A list of phases that match.
     */
    @GET
    @Path("/byTitle")
    @PublicApiMember
    List<Phase> searchPhasesByTitle(@QueryParam("phaseTitle") String phaseTitle, @QueryParam("releaseId") String releaseId);

    /**
     * Search phase within a release.
     * <p>
     * ReleaseId is required, all other parameters are optional. Providing the optional parameters will narrow down search.
     *
     * @param phaseTitle   (part of) the phase title (optionally, if not provided, all phases are returned)
     * @param releaseId    the release identifier.
     * @param phaseVersion phase version (optionally, if not provided, no filtering will be applied)
     * @return A list of phases that match.
     */
    @GET
    @Path("/search")
    @PublicApiMember
    List<Phase> searchPhases(@QueryParam("phaseTitle") String phaseTitle,
                             @QueryParam("releaseId") String releaseId,
                             @QueryParam("phaseVersion") PhaseVersion phaseVersion);


    /**
     * Adds a phase to a release. If no position is specified, then the phase is added as the last phase.
     * All nested configuration items (e.g. tasks) will be removed from the provided phase.
     *
     * @param releaseId the identifier of a template/release.
     * @param phase     the phase to add
     * @param position  the position in the template, 0-based. If left empty, then phase will the last phase in the template
     * @return add <code>Phase</code> object.
     */
    @POST
    @Path("/{releaseId:.*Release[^/]*}/phase")
    @PublicApiMember
    Phase addPhase(@PathParam("releaseId") String releaseId, Phase phase, @QueryParam("position") Integer position);

    /**
     * Copy a phase in a release.
     *
     * @param phaseIdToCopy  the phase id to copy
     * @param targetPosition the position where to copy the phase to within a release. Cannot be {@code < 0} and also not {@code >= release.phases.size()}
     * @return copy of a <code>Phase</code> object.
     */
    @POST
    @Path("/{phaseId:.*/Phase[^/]*}/copy")
    @PublicApiMember
    Phase copyPhase(@PathParam("phaseId") String phaseIdToCopy, @QueryParam("targetPosition") int targetPosition);

    /**
     * Wrapper method to add a phase to the release.
     *
     * @param releaseId the release id to add the phase to
     * @param phase     the phase to add
     * @return the phase object.
     */
    @PublicApiMember
    Phase addPhase(String releaseId, Phase phase);

    /**
     * Factory method to create a new instance of a phase.
     *
     * @return the task object.
     */
    @PublicApiMember
    Phase newPhase();

    /**
     * Factory method to create a new instance of a Phase object.
     * <p>
     * Example script which adds a custom phase to the current release:
     * <pre>
     * {@code
     * release = getCurrentRelease()
     * phase = phaseApi.newPhase("My New Phase")
     * phaseApi.addPhase(release.id, phase)
     * }
     * </pre>
     *
     * @param title the title of the phase.
     * @return the phase object.
     */
    @PublicApiMember
    Phase newPhase(String title);

    /**
     * Deletes the phase with the given ID.
     *
     * @param phaseId the phase identifier.
     */
    @DELETE
    @Path("/{phaseId:.*/Phase[^/]*}")
    @PublicApiMember
    void deletePhase(@PathParam("phaseId") String phaseId);
}
