package com.xebialabs.xlrelease.api.v1;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.apache.commons.fileupload.FileUploadException;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.CopyTemplate;
import com.xebialabs.xlrelease.api.v1.forms.CreateRelease;
import com.xebialabs.xlrelease.api.v1.forms.StartRelease;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.rest.AllCILevels;
import com.xebialabs.xlrelease.views.ImportResult;

/**
 * Operations on templates.
 */
@Path("/api/v1/templates")
@AllCILevels
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface TemplateApi extends ApiService {

    String SERVICE_NAME = "templateApi";

    default String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Returns the list of templates that are visible to the current user.
     *
     * @param title          an optional search filter containing the title of the template.
     * @param tags           an optional search filter containing list of template tags.
     * @param page           the page of results to return. Default value is 0.
     * @param resultsPerPage the number of results per page. Default value is 100. Maximum value is 100.
     * @param depth          the depth to search for. Default value is 1.
     * @return a list of release templates.
     */
    @GET
    @PublicApiMember
    List<Release> getTemplates(@QueryParam("title") String title,
                               @QueryParam("tag") List<String> tags,
                               @DefaultValue("0") @QueryParam(PAGE) Long page,
                               @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage,
                               @DefaultValue("1") @QueryParam(DEPTH) Integer depth);

    /**
     * Returns the list of the first 100 templates that are visible to the current user with depth 1.
     *
     * @param filter An optional search filter containing the title of the template
     * @deprecated Please use {@link #getTemplates(String, List, Long, Long, Integer)}
     */
    @Deprecated
    @PublicApiMember
    List<Release> getTemplates(String filter);

    /**
     * Updates the properties of a template.
     *
     * @param templateId the template identifier.
     * @param template   new contents of the template.
     * @return the updated release template.
     */
    @PUT
    @Path("/{templateId:.*Release[^/]*}")
    @PublicApiMember
    Release updateTemplate(@PathParam("templateId") String templateId, Release template);


    /**
     * Updates the properties of a template.
     *
     * @param template new contents of the template.
     * @return the updated release template.
     */
    @PublicApiMember
    default Release updateTemplate(Release template) {
        return updateTemplate(template.getId(), template);
    }

    /**
     * Creates a template inside a folder.
     * <p>
     * All nested configuration items (e.g. phases, tasks, teams) will be removed from the provided template, you can use
     * corresponding phaseApi or taskApi for those.
     *
     * @param template the contents of the release template.
     * @param folderId an optional parameter to create a template inside a folder
     * @return the created template.
     */
    @POST
    @PublicApiMember
    Release createTemplate(Release template, @QueryParam("folderId") String folderId);

    /**
     * Imports a template, serialized in JSON format.
     *
     * @param json     JSON representation of the template.
     * @param folderId the target folder.
     * @param version  the version of the template. (optional)
     * @return the newly created release template
     */
    @POST
    @Path("/import")
    @Consumes({MediaType.APPLICATION_JSON})
    @PublicApiMember
    List<ImportResult> importTemplate(String json, @QueryParam("folderId") String folderId, @QueryParam("version") String version);

    /**
     * Imports a template, serialized in JSON format.
     *
     * @param json     JSON representation of the template.
     * @param folderId the target folder.
     * @return the newly created release template
     */
    @PublicApiMember
    List<ImportResult> importTemplate(String json, String folderId);

    /**
     * Imports a template, serialized in XLR format, or a Releasefile (Releasefile.groovy) inside a .zip file.
     *
     * @param request the HttpServletRequest
     * @return the newly created release template
     */
    @POST
    @Path("/import")
    @Consumes({MediaType.MULTIPART_FORM_DATA})
    List<ImportResult> importTemplateAsXlr(@Context HttpServletRequest request,
                                           @QueryParam("folderId") String folderId)
            throws IOException, FileUploadException;

    /**
     * Returns the template for the given identifier.
     *
     * @param templateId the template identifier.
     * @return the release template.
     */
    @GET
    @Path("/{templateId:.*Release[^/]*}")
    @PublicApiMember
    Release getTemplate(@PathParam("templateId") String templateId);

    /**
     * Deletes a template.
     *
     * @param templateId the template identifier
     */
    @DELETE
    @Path("/{templateId:.*Release[^/]*}")
    @PublicApiMember
    void deleteTemplate(@PathParam("templateId") String templateId);

    /**
     * Creates a release from a template.
     *
     * @param templateId    the template identifier of the template the release is based on.
     * @param createRelease the parameters for the new release.
     * @return the newly created release.
     */
    @POST
    @Path("/{templateId:.*Release[^/]*}/create")
    @PublicApiMember
    Release create(@PathParam("templateId") String templateId, CreateRelease createRelease);

    /**
     * Starts a release from a template.
     *
     * @param templateId   the template identifier of the template the release is based on.
     * @param startRelease the parameters for the new release.
     * @return the newly created and started release.
     */
    @POST
    @Path("/{templateId:.*Release[^/]*}/start")
    @PublicApiMember
    Release start(@PathParam("templateId") String templateId, StartRelease startRelease);

    /**
     * Get variables.
     *
     * @param templateId the identifier of the release or template.
     * @return a list of variables.
     */
    @GET
    @Path("/{templateId:.*Release[^/]*}/variables")
    @PublicApiMember
    List<Variable> getVariables(@PathParam("templateId") String templateId);

    /**
     * Returns the variable for the given identifier.
     *
     * @param variableId the identifier of the variable.
     * @return the variable.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}")
    @PublicApiMember
    Variable getVariable(@PathParam("variableId") String variableId);

    /**
     * Returns possible values for the variable with the given identifier.
     *
     * @param variableId the variable identifier.
     * @return possible values for the variable.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}/possibleValues")
    @PublicApiMember
    Collection<Object> getVariablePossibleValues(@PathParam("variableId") String variableId);

    /**
     * Returns true if variable with the given identifier is used in the template.
     *
     * @param variableId the variable ID.
     * @return true if variable with the given identifier is used in the template.
     */
    @GET
    @Path("/{variableId:.*/Variable[^/]*}/used")
    @PublicApiMember
    Boolean isVariableUsed(@PathParam("variableId") String variableId);


    /**
     * Replace variable occurrences with the given replacement.
     *
     * @param variableId      the variable ID.
     * @param variableOrValue an object with a variable or value replacing the initial variable.
     */
    @POST
    @Path("/{variableId:.*/Variable[^/]*}/replace")
    @PublicApiMember
    void replaceVariable(@PathParam("variableId") String variableId, VariableOrValue variableOrValue);

    /**
     * Delete variable from template.
     *
     * @param variableId the variable ID.
     */
    @DELETE
    @Path("/{variableId:.*/Variable[^/]*}")
    @PublicApiMember
    void deleteVariable(@PathParam("variableId") String variableId);

    /**
     * Create a new variable.
     *
     * @param templateId the identifier of a template.
     * @param variable   the <code>Variable</code> object describing the new variable.
     * @return created <code>Variable</code> object.
     */
    @POST
    @Path("/{templateId:.*Release[^/]*}/variables")
    @PublicApiMember
    Variable createVariable(@PathParam("templateId") String templateId, com.xebialabs.xlrelease.api.v1.forms.Variable variable);

    /**
     * Update the variable list.
     *
     * @param releaseId the identifier of the release.
     * @param variables the variable list to update.
     * @return updated variables.
     */
    @PUT
    @Path("/{releaseId:.*Release[^/]*}/variables")
    @PublicApiMember
    List<Variable> updateVariables(@PathParam("releaseId") String releaseId, List<Variable> variables);

    @PUT
    @Path("/{variableId:.*/Variable[^/]*}")
    Variable updateVariable(@PathParam("variableId") String variableId, Variable variable);

    /**
     * Updates the properties of a variable.
     *
     * @param variable new contents of the variable.
     * @return the updated variable.
     */
    @PublicApiMember
    default Variable updateVariable(Variable variable) {
        return updateVariable(variable.getId(), variable);
    }

    /**
     * Returns possible permissions.
     *
     * @return list of permissions for templates and releases
     */
    @GET
    @Path("/permissions")
    @PublicApiMember
    List<String> getPermissions();

    /**
     * Returns effective teams of the template.
     *
     * @param templateId the identifier of the template
     * @return a list of effective teams
     */
    @GET
    @Path("/{templateId:.*Release[^/]*}/teams")
    @PublicApiMember
    List<TeamView> getTeams(@PathParam("templateId") String templateId);

    /**
     * Sets teams of the template.
     *
     * @param templateId the identifier of the template
     * @return a list of updated teams
     */
    @POST
    @Path("/{templateId:.*Release[^/]*}/teams")
    @PublicApiMember
    List<TeamView> setTeams(@PathParam("templateId") String templateId, List<TeamView> teamDtos);

    /**
     * Exports template as a zip download, in xlr format
     *
     * @param templateId the identifier of the template
     * @return zip archive of template
     */
    @GET
    @Path("zip/{templateId:.*Release[^/]*}")
    @PublicApiMember
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    Response exportTemplateToZip(@PathParam("templateId") @IdParam String templateId);

    /**
     * Makes a copy of the template on the current folder
     *
     * @param templateId:   the full templateID: Applications/FolderXXXX/ReleaseYYYY
     * @param copyTemplate: new properties of the template
     * @return the new template
     * @since 10.0
     */
    @POST
    @Path("/{templateId:.*Release[^/]*}/copy")
    Release copyTemplate(@PathParam("templateId") String templateId, CopyTemplate copyTemplate);

    /**
     * Makes a copy of the template on the current folder
     *
     * @param templateId:  the full templateID: Applications/FolderXXXX/ReleaseYYYY
     * @param title:       the new template title
     * @param description: the new template description
     * @return the new template
     * @since 10.0
     */
    @PublicApiMember
    default Release copyTemplate(String templateId, String title, String description) {
        final CopyTemplate copyTemplate = new CopyTemplate();
        copyTemplate.setTitle(title);
        copyTemplate.setDescription(description);
        return copyTemplate(templateId, copyTemplate);
    }

    @GET
    @Path("logo/{logoId:.*/TemplateLogo[^/]*}")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    Response downloadTemplateLogo(@PathParam("logoId") String logoId) throws IOException;

}
