package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.springframework.data.domain.Page;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.filter.CategoryFilters;
import com.xebialabs.xlrelease.api.v1.forms.CategoryForm;
import com.xebialabs.xlrelease.domain.Category;
import com.xebialabs.xlrelease.rest.AllCILevels;

/**
 * Operations on categories.
 */
@Path("/api/v1/categories")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@AllCILevels
@PublicApi
@ShowOnlyPublicApiMembers
public interface CategoryApi extends ApiService {
    String SERVICE_NAME = "categoryApi";

    default String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Searches categories by filters
     *
     * @param categoryFilters the query parameters derived from CategoryFilters property names used as a search criteria
     * @param page            the page of results to return. Default value is 0.
     * @param resultsPerPage  the number of results per page. Default and maximum value is 100.
     * @return the list of matching categories
     */
    @GET
    Page<Category> searchCategoriesPage(@BeanParam CategoryFilters categoryFilters,
                                        @DefaultValue("0") @QueryParam(PAGE) int page,
                                        @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage);

    /**
     * Gets List of categories
     *
     * @param categoryFilters the search criteria
     * @param page            the page of results to return. Default value is 0.
     * @param resultsPerPage  the number of results per page. Default and maximum value is 100.
     * @return found {@code List<Category>} object
     */
    @PublicApiMember
    default List<Category> searchCategories(CategoryFilters categoryFilters, int page, int resultsPerPage) {
        return searchCategoriesPage(categoryFilters, page, resultsPerPage).getContent();
    }

    /**
     * Create new category
     *
     * @param categoryForm  the content of the category
     * @return created category
     */
    @POST
    @PublicApiMember
    Category addCategory(CategoryForm categoryForm);

    /**
     * Updates the category
     *
     * @param ciUid  the identifier of the category
     * @param category  new content of the category
     * @return the updated category
     */
    @PUT
    @PublicApiMember
    @Path("{ciUid:.*}")
    Category updateCategory(@PathParam("ciUid") Integer ciUid, Category category);

    /**
     * Deletes a category
     *
     * @param ciUid the identifies of the category
     */
    @DELETE
    @PublicApiMember
    @Path("{ciUid:.*}")
    void deleteCategory(@PathParam("ciUid") Integer ciUid);
}
