package com.xebialabs.xlrelease.api.v1;

import java.io.IOException;
import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters;
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord;
import com.xebialabs.xlrelease.param.IdParam;

/**
 * Operations on report.
 */
@Path("/api/v1/reports")
@Produces({MediaType.APPLICATION_JSON})
@Consumes({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface ReportApi extends ApiService {

    default String serviceName() {
        return "reportApi";
    }

    /**
     * Returns the list of reporting records associated with release.
     *
     * @param releaseId the identifier of the release.
     * @return found {@code List<TaskReportingRecord>} object
     */
    @GET
    @Path("/records/{releaseId:.*Release[^/]*}")
    @PublicApiMember
    List<TaskReportingRecord> getRecordsForRelease(@PathParam("releaseId") String releaseId);

    /**
     * Returns the list of reporting records associated with task.
     *
     * @param taskId the identifier of the task.
     * @return found {@code List<TaskReportingRecord>} object
     */
    @GET
    @Path("/records/{taskId:.*/Task[^/]*}")
    @PublicApiMember
    List<TaskReportingRecord> getRecordsForTask(@PathParam("taskId") String taskId);

    /**
     * Searches records.
     *
     * @param facetFilters the search criteria
     * @return found {@code List<TaskReportingRecord>} object
     */
    @POST
    @Path("/records/search")
    @PublicApiMember
    List<TaskReportingRecord> searchRecords(FacetFilters facetFilters);

    /**
     * Download the single-release report by type.
     *
     * @param reportType the type of report
     * @param releaseId  the identifier of the release
     * @return found {@code List<TaskReportingRecord>} object
     */
    @GET
    @Path("/download/{reportType}/{releaseId:.*Release[^/]*}")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    Response downloadReleaseReport(@PathParam("reportType") String reportType, @PathParam("releaseId") @IdParam String releaseId);

    /**
     * Returns the single-release report file by type.
     *
     * @param reportType the type of report
     * @param releaseId  the identifier of the release
     * @return the attachment file.
     */
    @PublicApiMember
    byte[] getReleaseReport(String reportType, String releaseId) throws IOException;
}
