package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.rest.AllCILevels;
import com.xebialabs.xlrelease.risk.configuration.RiskGlobalThresholds;
import com.xebialabs.xlrelease.risk.domain.Risk;
import com.xebialabs.xlrelease.risk.domain.RiskProfile;
import com.xebialabs.xlrelease.risk.domain.riskassessors.RiskAssessor;

/**
 * Operations on risk.
 */
@Path("/api/v1/risks")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
@PublicApi
@ShowOnlyPublicApiMembers
public interface RiskApi extends ApiService {

    default String serviceName() {
        return "riskApi";
    }

    /**
     * Returns the release risk score.
     *
     * @param riskId the full identifier of  release risk score.
     * @return risk.
     */
    @GET
    @Path("/{riskId:.*/Risk}")
    @AllCILevels
    Risk getRisk(@PathParam("riskId") String riskId);

    /**
     * Returns the release risk score.
     *
     * @param release the release from which you want risk score.
     * @return risk.
     */
    @PublicApiMember
    Risk getRisk(final Release release);

    /**
     * Returns the global risk thresholds configuration.
     *
     * @return global risk thresholds.
     */
    @GET
    @Path("/config")
    @AllCILevels
    @PublicApiMember
    RiskGlobalThresholds getRiskGlobalThresholds();

    /**
     * Update the global risk thresholds configuration.
     *
     * @param thresholds the updated global risk thresholds.
     * @return global risk thresholds.
     */
    @PUT
    @Path("/config")
    @AllCILevels
    @PublicApiMember
    RiskGlobalThresholds updateRiskGlobalThresholds(final RiskGlobalThresholds thresholds);

    /**
     * Returns list of all risk profile configurations.
     *
     * @return risk profiles list
     */
    @GET
    @Path("/profiles")
    @PublicApiMember
    List<RiskProfile> getRiskProfiles();

    /**
     * Returns the risk profile for the given identifier.
     *
     * @param riskProfileId the full identifier of risk profile.
     * @return risk profile.
     */
    @GET
    @Path("/profiles/{riskProfileId:.*/RiskProfile[^/]*|new}")
    @PublicApiMember
    RiskProfile getRiskProfile(@PathParam("riskProfileId") String riskProfileId);

    /**
     * Returns the risk profile with the given title.
     *
     * @param title title of the risk profile.
     * @return risk profile.
     */
    @PublicApiMember
    RiskProfile getRiskProfileByTitle(String title);

    /**
     * Update the risk profile.
     *
     * @param riskProfileId the full identifier of risk profile.
     * @param riskProfile   the updated risk profile.
     * @return the updated risk profile.
     */
    @PUT
    @Path("/profiles/{riskProfileId:.*/RiskProfile[^/]*}")
    RiskProfile updateRiskProfile(@PathParam("riskProfileId") String riskProfileId, RiskProfile riskProfile);

    /**
     * Update the risk profile.
     *
     * @param riskProfile the updated risk profile.
     * @return the updated risk profile.
     */
    @PublicApiMember
    RiskProfile updateRiskProfile(final RiskProfile riskProfile);

    /**
     * Create risk profile.
     *
     * @param riskProfile the new risk profile.
     * @return risk profile.
     */
    @POST
    @Path("/profiles")
    @PublicApiMember
    RiskProfile createRiskProfile(RiskProfile riskProfile);

    /**
     * Delete risk profile.
     *
     * @param riskProfileId the full identifier of risk profile.
     */
    @DELETE
    @Path("/profiles/{riskProfileId:.*/RiskProfile[^/]*}")
    void deleteRiskProfile(@PathParam("riskProfileId") String riskProfileId);

    /**
     * Delete risk profile.
     *
     * @param riskProfile the risk profile you would like to delete.
     */
    @PublicApiMember
    void deleteRiskProfile(final RiskProfile riskProfile);

    /**
     * Make copy of risk profile with given id.
     *
     * @param riskProfileId the full identifier of risk profile.
     * @return risk profile.
     */
    @POST
    @Path("/profiles/{riskProfileId:.*/RiskProfile[^/]*}/copy")
    RiskProfile copyRiskProfile(@PathParam("riskProfileId") String riskProfileId);

    /**
     * Make copy of given risk profile.
     *
     * @param riskProfile the risk profile from which a copy will be created.
     * @return risk profile.
     */
    @PublicApiMember
    RiskProfile copyRiskProfile(final RiskProfile riskProfile);

    /**
     * Returns the list of all risk assessor.
     *
     * @return list of all risk assessor.
     */
    @GET
    @Path("/assessors")
    @AllCILevels
    @PublicApiMember
    List<RiskAssessor> getAllRiskAssessors();
}
