#
# Copyright (c) 2020. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#


import json
from com.xebialabs.xlrelease.plugin.vault import VaultServer
from com.xebialabs.xlrelease.plugin.vault import HttpRequestHelper


class VaultClient(object):
    # TODO: Consider adding support for headers
    # TODO: Consider adding support for namespace
    # TODO: Consider adding support for TTL
    # TODO: May need to consider adding/handling redirects (allow_redirects)

    # It is convenient to have separate values for exit codes, instead of the usual exit(1)
    VAULT_SERVER_SEALED = 1
    VAULT_NOT_AUTHENTICATED = 2
    VAULT_NO_TOKEN = 3
    VAULT_NO_SERVER_PROVIDED = 4
    VAULT_NOT_INITIALIZED = 5

    def __init__(self, url, token=None, username=None, password=None, logger=None, namespace=None, authenticationMethod=None, role_id=None, secret_id=None):
        self.httpRequestHelper = HttpRequestHelper(None,0,None,None)
        self.logger = logger
        self.logger.info("=== VAULT START INITIALIZE ===")
        self.host = url
        self.usertoken = None
        self.token = None
        if not self.host.endswith("/"):
            self.host = "%s/" % self.host
        self.authenticationMethod = authenticationMethod
        if self.host is None:
            self.exit(self.VAULT_NO_SERVER_PROVIDED, "No Server was provided")
        if str(self.authenticationMethod) != "PAT" and  str(self.authenticationMethod) != "None":
            self.username = username
            self.password = password
            self.namespace = namespace
            self.role_id = role_id
            self.secret_id = secret_id
            self.urlwithusername = ""
            if self.password is not None and self.username is not None:
                passworddict = {'password': self.password}
                app_json = json.dumps(passworddict)
                self.urlwithusername = self.host+'v1/auth/userpass/login/'+self.username
            if str(self.authenticationMethod) == str(VaultServer.AuthenticationMethod.LDAP):
                self.urlwithusername = self.host+'v1/auth/ldap/login/'+self.username
            elif str(self.authenticationMethod) == str(VaultServer.AuthenticationMethod.Approle):
                self.urlwithusername = self.host+'v1/auth/approle/login'
                appRoledict = {'role_id': self.role_id,'secret_id': self.secret_id}
                app_json = json.dumps(appRoledict)
            #String url, String method, String jsonHeaders, String body, String filePath
            response = self.httpRequestHelper.executeRequest(self.urlwithusername,"POST",None,app_json,None)
            responseload = json.loads(response['response'])
            statuscode = json.loads(response['statusCode'])
            if int(statuscode) > 300:
                raise Exception("Unable to get User Token")
            else:
                self.usertoken = responseload['auth']['client_token']
                self.client = None
        elif str(self.authenticationMethod) == "PAT":
            self.token = token
            self.namespace = namespace
            self.client = None
        elif str(self.authenticationMethod) == "None":
            self.client = None
            self.token = None
            self.namespace = namespace
        else:
            exit("VAULT ({}) {}".format(self.VAULT_NO_TOKEN, "No Token or Username/Password/role_id-secret_id provided"))

        self.logger.info("=== VAULT END INITIALIZE ===")

    @staticmethod
    def exit (code, message, authenticationType=None):
        print ("Exit", message)
        if authenticationType is not None:
            if authenticationType is VaultServer.AuthenticationMethod.Basic or authenticationType is VaultClient.AuthenticationMethod.LDAP or authenticationType is VaultClient.AuthenticationMethod.Approle:
                VaultClient.revokeUserPassToken()
        exit("VAULT ({}) {}".format(code, message))

    def revokeUserPassToken(self):
        tokendict = {'token': self.usertoken}
        headersdict = {'X-Vault-Token': self.usertoken}
        self.urlRevoke = self.host+'v1/auth/token/revoke'
        responseObj = self.httpRequestHelper.executeRequest(self.urlRevoke,"POST",json.dumps(headersdict),json.dumps(tokendict),None)


    # TODO: I am not sure we need this.  The operation is simple and we can inline the conditional elsewhere.
    def get_headers(self):
        headersdict = None
        if self.usertoken is not None:
            headersdict = {'X-Vault-Token': self.usertoken}
        else:
            if self.token is not None:
                headersdict = {'X-Vault-Token': self.token}
        if self.namespace is not None:
            headersdict = {'X-Vault-Namespace': self.namespace}
        return headersdict

    def is_initialized(self):
        headers = self.get_headers()
        initurl = self.host+ 'v1/sys/init'
        stringHeaders = json.dumps(headers)
        responseObj = self.httpRequestHelper.executeRequest(initurl,"GET",stringHeaders,None,None)
        getresponseLoad = json.loads(responseObj['response'])
        if getresponseLoad["initialized"]:
            return True
        else:
            return False

    def is_sealed(self):
        headers = self.get_headers()
        sealurl = self.host+ 'v1/sys/seal-status'
        responseObj = self.httpRequestHelper.executeRequest(sealurl,"GET",json.dumps(headers),None,None)
        getresponseLoad = json.loads(responseObj['response'])
        if getresponseLoad["sealed"]:
            return True
        else:
            return False

    def is_authenticated(self):
        headers = self.get_headers()
        sealurl = self.host+ 'v1/sys/auth'
        responseObj = self.httpRequestHelper.executeRequest(sealurl,"GET",json.dumps(headers),None,None)
        getresponseLoad = json.loads(responseObj['response'])
        if 'data' in getresponseLoad:
            if getresponseLoad["data"]:
                return True
            else:
                return False
        else:
            exit("Vault Error: Check the Credentials Used")

    def read_secret(self, path):
        headers = self.get_headers()
        readv1url = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(readv1url,"GET",json.dumps(headers),None,None)
        getresponseLoad = json.loads(responseObj['response'])
        return getresponseLoad

    def create_secret(self, path, secret):
        headers = self.get_headers()
        createv1url = self.host+ path
        self.httpRequestHelper.executeRequest(createv1url,"POST",json.dumps(headers),json.dumps(secret),None)

    def create_secret_v2(self, path, secret):
        headers = self.get_headers()
        createv1url = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(createv1url,"POST",json.dumps(headers),json.dumps(secret),None)
        return json.loads(responseObj['response'])

    def update_metadata_v2(self, path, secret):
        headers = self.get_headers()
        createv1url = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(createv1url,"POST",json.dumps(headers),json.dumps(secret),None)
        return json.loads(responseObj['response'])

    def patch_secret(self, path, secret):
        headers = self.get_headers()
        createv1url = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(createv1url,"PATCH",json.dumps(headers),json.dumps(secret),None)
        return json.loads(responseObj['response'])

    def delete_secret(self, path):
        headers = self.get_headers()
        deletev1url = self.host+ path
        self.httpRequestHelper.executeRequest(deletev1url,"DELETE",json.dumps(headers),None,None)

    # Used for both Delete and Undelete Secret Version in V2
    def delete_secret_version(self, path, versions):
        headers = self.get_headers()
        deleteVersionurl = self.host+ path
        self.httpRequestHelper.executeRequest(deleteVersionurl,"POST",json.dumps(headers),json.dumps(versions),None)

    def delete_metadata_all_version(self, path):
        headers = self.get_headers()
        delete_metadata_all_versionnurl = self.host+ path
        self.httpRequestHelper.executeRequest(delete_metadata_all_versionnurl,"DELETE",json.dumps(headers),None,None)

    # Used for both Delete and Undelete Secret Version in V2
    def destroy_secret_version(self, path, versions):
        headers = self.get_headers()
        destroyVersionurl = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(destroyVersionurl,"PUT",json.dumps(headers),json.dumps(versions),None)
        return json.loads(responseObj['response'])

    def list_secret_v2(self, path):
        headers = self.get_headers()
        destroyVersionurl = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(destroyVersionurl,"LIST",json.dumps(headers),None,None)
        return json.loads(responseObj['response'])

    def enable_engine(self, path, payload):
        headers = self.get_headers()
        enableurl = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(enableurl,"POST",json.dumps(headers),json.dumps(payload),None)

    def list_secrets(self, path):
        headers = self.get_headers()
        readv1url = self.host+ path
        response = self.httpRequestHelper.executeRequest(readv1url,"LIST",json.dumps(headers),None,None)
        getresponseLoad = json.loads(response['response'])
        return getresponseLoad

    def configure_kv_engine(self, path, body):
        headers = self.get_headers()
        configureurl = self.host+ path
        self.httpRequestHelper.executeRequest(configureurl,"POST",json.dumps(headers),json.dumps(body),None)

    def get_config_kv_engine(self, path):
        headers = self.get_headers()
        configureurl = self.host+ path
        response = self.httpRequestHelper.executeRequest(configureurl,"GET",json.dumps(headers),None,None)
        getresponseLoad = json.loads(response['response'])
        return getresponseLoad

    # List Secrets
    def list_secret_v2(self, path):
        headers = self.get_headers()
        destroyVersionurl = self.host+ path
        responseObj = self.httpRequestHelper.executeRequest(destroyVersionurl,"LIST",json.dumps(headers),None,None)
        print(responseObj)
        return json.loads(responseObj['response'])