import React, { useEffect } from 'react';
import { DotAvatar, DotIcon, DotTypography } from '@digital-ai/dot-components';
import { ButtonWithLink, CommonCardTileComponent } from '../../CommonCardTile/common-card-tile.component';
import { ActiveApplicationTile } from './active-application-tile.component';
import { applicationTile, getApplicationTileState } from './ducks/application-tile.reducer';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { ActiveTileSkeleton } from '../../CommonCardTile/active-tile.skeleton';
import { workflow } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { DEFAULT_APPLICATION_SEARCH_CATEGORY } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/constants';

const { loadApplicationTileData, setApplicationDialogOpen } = applicationTile.actions;
const { init } = workflow.actions;

const applicationImgSrc = 'static/23.3.3/styles/img/application_empty_tile_sample.svg';

export interface ApplicationTileProps {
    description: string;
    title: string;
}

export const ApplicationTileComponent = ({ description, title }: ApplicationTileProps) => {
    const dispatch = useAppDispatch();

    const { applicationTileData, isLoading } = useAppSelector(getApplicationTileState);
    const showActiveTile = applicationTileData.applications.length > 0;

    useEffect(() => {
        dispatch(loadApplicationTileData());
    }, []);

    const handleAddApplicationClick = () => {
        dispatch(init({ categories: [DEFAULT_APPLICATION_SEARCH_CATEGORY] }));
    };

    const handleViewApplicationsClick = () => {
        dispatch(setApplicationDialogOpen(true));
    };

    const primaryButton: ButtonWithLink = {
        buttonCaption: 'Add application',
        onClick: handleAddApplicationClick,
        buttonType: showActiveTile ? 'text' : 'outlined',
    };

    const secondaryButton: ButtonWithLink = {
        buttonCaption: 'View applications',
        onClick: handleViewApplicationsClick,
        buttonType: 'outlined',
    };

    const tileHeaderEndContent = (
        <>
            <DotAvatar alt="total-applications" iconId="apps" type="icon" />
            <div>
                <DotTypography variant="subtitle1">{applicationTileData.total}</DotTypography>
                <DotTypography className="header-tile-subtitle" variant="body2">
                    Total applications
                </DotTypography>
            </div>
            {showActiveTile && <DotIcon iconId="circle-info-outline" tooltip={description} />}
        </>
    );

    const renderTile = () =>
        showActiveTile ? (
            <ActiveApplicationTile
                activeTileData={applicationTileData}
                headerEndContent={tileHeaderEndContent}
                primaryButton={primaryButton}
                secondaryButton={secondaryButton}
                title={title}
            />
        ) : (
            <CommonCardTileComponent
                description={description}
                headerEndContent={applicationTileData.total > 0 ? tileHeaderEndContent : undefined}
                imageLink={applicationImgSrc}
                isMarkdown={true}
                primaryButton={primaryButton}
                secondaryButton={showActiveTile ? secondaryButton : undefined}
                title={title}
            />
        );

    return isLoading ? <ActiveTileSkeleton /> : renderTile();
};
