import { calculateEmptyDataInfo, calculateTableColumns, calculateWorkflowTileSubtitle, mapToHomeWorkflow } from './helpers';
import { WorkflowTileToggleEnum } from '../workflow-tile.component';
import { Workflow } from '../../../../../../../../../../../core/xlr-ui/app/types';

describe('workflow tile helper functions', () => {
    describe('calculateTableColumns', () => {
        const defaultColumns = [
            { id: 'workflowName', label: 'Workflow', truncate: true },
            { id: 'numberOfExecutions', label: 'No. of executions', width: '152px' },
            { id: 'category', label: 'Category', width: '37%' },
        ];

        it('should return correct array of columns for workflow table', () => {
            expect(calculateTableColumns(WorkflowTileToggleEnum.MostPopular)).toStrictEqual(defaultColumns);

            expect(calculateTableColumns(WorkflowTileToggleEnum.Running)).toStrictEqual([
                { id: 'workflowName', label: 'Workflow', truncate: true },
                { id: 'status', label: 'Status', width: '136px' },
                { id: 'started', label: 'Started', width: '136px' },
                { id: 'category', label: 'Category', width: '37%' },
            ]);

            expect(calculateTableColumns(WorkflowTileToggleEnum.Recent)).toStrictEqual([
                { id: 'workflowName', label: 'Workflow', truncate: true },
                { id: 'status', label: 'Status', width: '136px' },
                { id: 'ended', label: 'Ended', width: '136px' },
                { id: 'category', label: 'Category', width: '37%' },
            ]);

            expect(calculateTableColumns(WorkflowTileToggleEnum.Overview)).toStrictEqual(defaultColumns);
        });
    });

    describe('calculateWorkflowTileSubtitle', () => {
        it('should return correct subtitle for active workflow tile', () => {
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.MostPopular)).toBe('Account workflows in Last 30 days - Top 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.MostPopular, 48)).toBe('Account workflows in Last 2 days - Top 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.MostPopular, 24)).toBe('Account workflows in Last day - Top 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.MostPopular, 49)).toBe('Account workflows in Last 2 days - Top 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.MostPopular, 1)).toBe('Account workflows in Last hour - Top 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.MostPopular, 0)).toBe('Account workflows in Last 0 hours - Top 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.Recent)).toBe('Workflow executions - Most recent 5');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.Running)).toBe('Running workflows - 5 oldest');
            expect(calculateWorkflowTileSubtitle(WorkflowTileToggleEnum.Overview)).toBe('');
        });
    });

    describe('calculateEmptyDataInfo', () => {
        it('should return correct empty data', () => {
            expect(calculateEmptyDataInfo(WorkflowTileToggleEnum.MostPopular)).toStrictEqual({
                imageLink: 'static/23.3.0/styles/img/supportdata_maintenance_monochromatic.svg',
                message: 'Information is not available.',
            });
            expect(calculateEmptyDataInfo(WorkflowTileToggleEnum.Recent)).toStrictEqual({
                imageLink: 'static/23.3.0/styles/img/timeline_monochromatic.svg',
                message: 'You do not have workflow executions.',
            });
            expect(calculateEmptyDataInfo(WorkflowTileToggleEnum.Running)).toStrictEqual({
                imageLink: 'static/23.3.0/styles/img/workflows_forms.svg',
                message: 'You do not have running workflows.',
            });
            expect(calculateEmptyDataInfo(WorkflowTileToggleEnum.Overview)).toStrictEqual({
                imageLink: '',
                message: '',
            });
        });
    });

    describe('mapToHomeWorkflow', () => {
        it('should return array of HomeWorkflow elements', () => {
            const workflows: Array<Workflow> = [
                {
                    allowTargetFolderOverride: false,
                    author: 'someone',
                    categories: ['Cloud'],
                    defaultTargetFolder: 'targetFolderName',
                    description: 'some description',
                    executions: 5,
                    folderId: 'folderId',
                    folderTitle: 'folderTitle',
                    id: 'workflowId',
                    logo: null,
                    scmTraceabilityData: {
                        commit: null,
                        remote: null,
                    },
                    tags: [],
                    title: 'Workflow title',
                },
            ];

            expect(mapToHomeWorkflow(workflows)).toStrictEqual([
                {
                    categories: ['Cloud'],
                    folderId: 'folderId',
                    folderPath: '',
                    logoId: null,
                    numberOfExecutions: 5,
                    releaseId: 'workflowId',
                    releaseTitle: 'Workflow title',
                },
            ]);
        });

        it('should return empty array', () => {
            expect(mapToHomeWorkflow([])).toStrictEqual([]);
        });
    });
});
