/* global document */
import React, {Component} from 'react';
import PropTypes from 'prop-types';
import ReactDOM from 'react-dom';

function clickOutside() {
  return WrappedComponent => class WithClickOutside extends Component {
    static get propTypes() {
      return {
        enable: PropTypes.bool,
        onClickOutside: PropTypes.func.isRequired,
      };
    }

    static get defaultProps() {
      return {
        enable: true,
      };
    }

    constructor(props) {
      super(props);
      this.mounted = true;
      this.handleDocumentClick = this.handleDocumentClick.bind(this);
    }

    componentDidMount() {
      document.addEventListener('click', this.handleDocumentClick, false);
      document.addEventListener('touchend', this.handleDocumentClick, false);
    }

    componentWillUnmount() {
      this.mounted = false;
      document.removeEventListener('click', this.handleDocumentClick, false);
      document.removeEventListener('touchend', this.handleDocumentClick, false);
    }

    handleDocumentClick(event) {
      if (this.mounted && this.props.enable) {
        /* eslint-disable react/no-find-dom-node */
        if (!ReactDOM.findDOMNode(this).contains(event.target)) {
          if (this.props.onClickOutside) {
            this.props.onClickOutside();
          }
        }
      }
    }

    render() {
      return (
        <WrappedComponent {...this.props} />
      );
    }
  };
}

export default clickOutside;
