import React from 'react';
import PropTypes from 'prop-types';
import colors from '../../../js/colors';

export default class EditableLabel extends React.Component {

    static propTypes = {
        disabled: PropTypes.bool,
        inputBorderWidth: PropTypes.string,
        inputClassName: PropTypes.string,
        inputFontSize: PropTypes.string,
        inputFontWeight: PropTypes.string,
        inputHeight: PropTypes.string,
        inputMaxLength: PropTypes.number,
        inputPlaceHolder: PropTypes.string,
        inputTabIndex: PropTypes.number,
        inputWidth: PropTypes.string,
        isEditing: PropTypes.bool,
        labelClassName: PropTypes.string,
        labelFontSize: PropTypes.string,
        labelFontWeight: PropTypes.string,
        onFocus: PropTypes.func,
        onFocusOut: PropTypes.func,
        onValidityChange: PropTypes.func,
        required: PropTypes.bool,
        text: PropTypes.string.isRequired,
        validators: PropTypes.arrayOf(PropTypes.func)
    };

    constructor(props) {
        super(props);

        this.state = {
            isEditing: this.props.isEditing || false,
            text: this.props.text || "",
            lastSaved: this.props.text,
            validationResult: {valid: true}
        };
    }

    rollback() {
        this.setState({
            text: this.state.lastSaved,
        });
        this.checkForValidation(this.state.lastSaved);
    }

    _handleFocus = () => {
        const {disabled, onFocus, onFocusOut} = this.props;
        if (!disabled) {
            const {isEditing, text} = this.state;
            if (isEditing) {
                if (text && text.length > 0) {
                    /* eslint-disable angular/typecheck-function */
                    if (typeof onFocusOut === 'function') {
                        onFocusOut(text, ::this.rollback);
                    }
                    /* eslint-enable */
                }
            } else {
                /* eslint-disable angular/typecheck-function */
                if (typeof onFocus === 'function') {
                    onFocus(text);
                }
                /* eslint-enable */
            }

            if (text && text.length > 0) {
                this.setState({
                    lastSaved: text,
                    isEditing: !isEditing,
                });
            }
        }
    };

    _selectValue = (e) => {
        e.target.select();
    };

    _handleChange = (ev) => {
        this.setState({
            text: ev.target.value,
        });
        this.checkForValidation(ev.target.value);
    };

    checkForValidation(value){
        if (this.props.validators) {
            const isValid = ({message}) => message.type !== 'ERROR';

            const validationToShow = this.props.validators
                .map((validator) => validator(value.trim()))
                .find((result) => result.condition);

            const validationResult = (validationToShow && validationToShow.condition) ?
                {
                    message: {...validationToShow.message},
                    valid: isValid(validationToShow)
                } :
                {valid: true};
            this.setState({validationResult});
            this.props.onValidityChange(validationResult.valid);
        }
    }

    _handleKeyDown = e => {
        switch (e.key) {
            case 'Enter':
                this._handleChange(e);
                this._handleFocus(e);
                break;
            case 'Escape':
                this.setState({
                    text: this.state.lastSaved,
                    isEditing: false
                });
                break;
        }
    };

    render() {
        return (
            <div>
                {this.state.isEditing ? (
                    <React.Fragment>
                        <input autoFocus
                               className={`${this.props.inputClassName} ${this.state.validationResult.valid ? '' : 'input-field-error'}`}
                               maxLength={this.props.inputMaxLength}
                               onBlur={this._handleFocus}
                               onChange={this._handleChange}
                               onFocus={this._selectValue}
                               onKeyDown={this._handleKeyDown}
                               placeholder={this.props.inputPlaceHolder}
                               style={{
                                   width: this.props.inputWidth,
                                   height: '30px',
                                   fontSize: this.props.inputFontSize,
                                   fontWeight: this.props.inputFontWeight,
                                   borderWidth: this.props.inputBorderWidth,
                                   borderColor: this.props.required && this.state.text.length === 0 ? colors.red : 'none'
                               }}
                               tabIndex={this.props.inputTabIndex}
                               type="text"
                               value={this.state.text}/>
                        {this.state.validationResult.valid ? <span/> : (
                            <span className={`validation-${this.state.validationResult.message.type.toLowerCase()}`}>
                                {this.state.validationResult.message.text}
                            </span>
                        )
                        }
                    </React.Fragment>
                ) : (
                    <label className={this.props.labelClassName}
                           onClick={this._handleFocus}
                           style={{
                               fontSize: this.props.labelFontSize,
                               fontWeight: this.props.labelFontWeight,
                               height: '30px'
                           }}
                           title={this.state.text}>
                        {this.state.text}
                    </label>
                )}
            </div>
        );
    }
}
